% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/nn2poly_methods.R
\name{predict.nn2poly}
\alias{predict.nn2poly}
\title{Predict method for \code{nn2poly} objects.}
\usage{
\method{predict}{nn2poly}(object, newdata, monomials = FALSE, layers = NULL, ...)
}
\arguments{
\item{object}{Object of class inheriting from 'nn2poly'.}

\item{newdata}{Input data as matrix, vector or dataframe.
Number of columns (or elements in vector) should be the number of variables
in the polynomial (dimension p). Response variable to be predicted should
not be included.}

\item{monomials}{Boolean determining if the returned item should contain the
evaluations of all the monomials of the provided polynomials
(\code{monomials==TRUE}), or if the final polynomial evaluation should be
computed, i.e., adding up all the monomials (\code{monomials==FALSE}).
Defaults to \code{FALSE}.}

\item{layers}{Vector containing the chosen layers from \code{object} to be
evaluated. If set to \code{NULL}, all layers are computed. Default is set
to \code{NULL}.}

\item{...}{Further arguments passed to or from other methods.}
}
\value{
Returns a matrix or list of matrices with the evaluation of each
polynomial at each layer as given by the provided \code{object} of class
\code{nn2poly}. The format can be as follows, depending on the layers
contained in \code{object} and the parameters \code{layers} and \code{monomials} values:
\itemize{
\item If \code{object} contains the polynomials of the last layer, as given by
\code{nn2poly(object, keep_layers = FALSE)}, then the output is:
\itemize{
\item A matrix: if \code{monomials==FALSE}, returns a matrix containing the
evaluation of the polynomials on the given data. The matrix has dimensions
\code{(n_sample, n_polynomials)}, meaning that each column corresponds to the
result of evaluating all the data for a polynomial. If a single polynomial is
provided, the output is a vector instead of a row matrix.
\item A 3D array: If \code{monomials==TRUE}, returns a 3D array containing the monomials of
each polynomial evaluated on the given data. The array has dimensions
\code{(n_sample, n_monomial_terms, n_polynomials)}, where element
\code{[i,j,k]} contains the evaluation on observation \code{i} on
monomial \code{j} of polynomial \code{k}, where monomial \code{j} corresponds
to the one on \code{poly$labels[[j]]}.
}
\item If \code{object} contains all the internal polynomials, as given by
\code{nn2poly(object, keep_layers = TRUE)}, then the output is a list of
layers (represented by \code{layer_i}), where each of them is another list with
\code{input} and \code{output} elements. Each of those elements contains the
corresponding evaluation of the "input" or "output" polynomial at the given layer,
as explained in the last layer case, which will be a matrix if \code{monomials==FALSE}
and a 3D array if \code{monomials==TRUE}.
}
}
\description{
Predicted values obtained with a \code{nn2poly} object on given data.
}
\details{
Internally uses \code{eval_poly()} to obtain the predictions. However, this only
works with a objects of class \code{nn2poly} while \code{eval_poly()} can be used
with a manually created polynomial in list form.

When \code{object} contains all the internal polynomials also, as given by
\code{nn2poly(object, keep_layers = TRUE)}, it is important to note that there
are two polynomial items per layer (input/output). These polynomial items will
also contain several polynomials of the same structure, one per neuron in the
layer, stored as matrix rows in \code{$values}. Please see the NN2Poly
original paper for more details.

Note also that "linear" layers will contain the same input and output results
as Taylor expansion is not used and thus the polynomials are also the same.
Because of this, in the situation of evaluating multiple layers we provide
the final layer with "input" and "output" even if they are the same, for
consistency.
}
\examples{
# Build a NN structure with random weights, with 2 (+ bias) inputs,
# 4 (+bias) neurons in the first hidden layer with "tanh" activation
# function, 4 (+bias) neurons in the second hidden layer with "softplus",
# and 1 "linear" output unit

weights_layer_1 <- matrix(rnorm(12), nrow = 3, ncol = 4)
weights_layer_2 <- matrix(rnorm(20), nrow = 5, ncol = 4)
weights_layer_3 <- matrix(rnorm(5), nrow = 5, ncol = 1)

# Set it as a list with activation functions as names
nn_object = list("tanh" = weights_layer_1,
                 "softplus" = weights_layer_2,
                 "linear" = weights_layer_3)

# Obtain the polynomial representation (order = 3) of that neural network
final_poly <- nn2poly(nn_object, max_order = 3)

# Define some new data, it can be vector, matrix or dataframe
newdata <- matrix(rnorm(10), ncol = 2, nrow = 5)

# Predict using the obtained polynomial
predict(object = final_poly, newdata = newdata)

# Predict the values of each monomial of the obtained polynomial
predict(object = final_poly, newdata = newdata, monomials = TRUE)

# Change the last layer to have 3 outputs (as in a multiclass classification)
# problem
weights_layer_4 <- matrix(rnorm(20), nrow = 5, ncol = 4)

# Set it as a list with activation functions as names
nn_object = list("tanh" = weights_layer_1,
                 "softplus" = weights_layer_2,
                 "linear" = weights_layer_4)

# Obtain the polynomial representation of that neural network
# Polynomial representation of each hidden neuron is given by
final_poly <- nn2poly(nn_object, max_order = 3, keep_layers = TRUE)

# Define some new data, it can be vector, matrix or dataframe
newdata <- matrix(rnorm(10), ncol = 2, nrow = 5)

# Predict using the obtained polynomials (for all layers)
predict(object = final_poly, newdata = newdata)

# Predict using the obtained polynomials (for chosen layers)
predict(object = final_poly, newdata = newdata, layers = c(2,3))

}
\seealso{
\code{\link[=nn2poly]{nn2poly()}}: function that obtains the \code{nn2poly} polynomial
object, \code{\link[=eval_poly]{eval_poly()}}: function that can evaluate polynomials in general,
\code{\link[stats:predict]{stats::predict()}}: generic predict function.
}
