% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/mod.run.R
\name{mod.run}
\alias{mod.run}
\title{Run population pharmacokinetic model with pre-defined search space}
\usage{
mod.run(
  string = NULL,
  dat = NULL,
  search.space = "ivbase",
  no.cores = NULL,
  penalty.control = penaltyControl(),
  param_table = NULL,
  nlmixr2autoinits = TRUE,
  reuse_cache = 1,
  precomputed_results_file = NULL,
  foldername = NULL,
  filename = "test",
  save_fit_rds = FALSE,
  save_csv = TRUE,
  .modEnv = NULL,
  verbose = TRUE,
  custom_config,
  ...
)
}
\arguments{
\item{string}{Numeric vector of parameter values. The length and interpretation
depends on the search.space configuration:
\itemize{
\item "ivbase": 10 values in order: (no.cmpt, eta.km, eta.vc,
eta.vp, eta.vp2, eta.q, eta.q2, mm, mcorr, rv)
\item "oralbase": 11 values in order: (no.cmpt, eta.km, eta.vc,
eta.vp, eta.vp2, eta.q, eta.q2,eta.ka, mm, mcorr, rv)
\item "custom": Length determined by custom_config$params in the order specified
}
The meaning of each element name is defined in \code{ppkmodGen()}.}

\item{dat}{A data frame containing pharmacokinetic data in standard
nlmixr2 format, including "ID", "TIME", "EVID", and "DV", and may include
additional columns.}

\item{search.space}{Character string specifying which search space to use.
Options are "ivbase", "oralbase", or "custom". Default is "ivbase".}

\item{no.cores}{Integer. Number of CPU cores to use. If NULL, uses
\code{rxode2::getRxThreads()}.}

\item{penalty.control}{A list of penalty control parameters defined by
\code{penaltyControl()}, specifying penalty values used for model diagnostics
during fitness evaluation.}

\item{param_table}{Optional data frame of initial parameter estimates. If NULL,
the table is generated by \code{auto_param_table()}.}

\item{nlmixr2autoinits}{Logical; if TRUE, use automatic initial estimates
from nlmixr2. Default is TRUE.}

\item{reuse_cache}{Integer; if 1, attempt to reuse cached results from
previous runs. Default is 1.}

\item{precomputed_results_file}{Character string; path to a CSV file containing
pre-computed model results for caching}

\item{foldername}{Character string specifying the folder name for storing
model files and results. If \code{NULL} (default), \code{tempdir()}
is used for temporary storage. If specified, a cache directory
is created in the current working directory.}

\item{filename}{Character string; base name for output files (without extension).
Required parameter with no default.}

\item{save_fit_rds}{Logical; if TRUE, save the fitted model object as an
RDS file. Default is FALSE.}

\item{save_csv}{Logical; if TRUE, save results to a CSV file. Default is
TRUE.}

\item{.modEnv}{Environment for storing state across multiple model runs. If
NULL, a new environment will be created.}

\item{verbose}{Logical; if TRUE, print progress messages. Default is
TRUE.}

\item{custom_config}{List; custom search space configuration for use with
search.space = "custom". Must contain four elements: route,
params, param_dependencies, and fixed_params. See Details
and Examples.}

\item{...}{Additional arguments passed to nlmixr2 control functions (e.g.,
saem.control, table.control, max_wall_time)}
}
\value{
Numeric value representing the fitness score of the fitted model
}
\description{
Fits a population PK model using nlmixr2 with configurable search spaces.
Supports pre-defined model structures (IV, oral) and custom configurations
for advanced modeling scenarios.
}
\details{
This function implements a flexible framework for fitting population PK models
with different structural configurations. It uses a configuration-driven approach
where the search.space parameter determines how the string vector
is interpreted and which model structure is generated.

\strong{Search Space Configurations:}
The function supports three types of search spaces:
\describe{
\item{ivbase}{Pre-defined IV bolus model with 11 parameters. Supports 1-3
compartment models with linear or Michaelis-Menten elimination.}
\item{oralbase}{Pre-defined oral model with 12 parameters (adds eta.ka for
first-order absorption). Same features as ivbase plus absorption kinetics.}
\item{custom}{User-defined model structure requiring custom_config
argument. Allows any combination of parameters supported by \code{ppkmodGen()}.
Supported parameters include: no.cmpt, abs.bio, abs.type, abs.delay,
eta.ka, eta.vc, eta.vp, eta.vp2, eta.q, eta.q2, mm, eta.km, eta.tlag, eta.n,
eta.mtt, eta.bio, eta.D2, eta.F1, eta.Fr, mcorr, rv, and allometric_scaling.
Note: eta.cl and eta.vmax are mutually exclusive and cannot be placed in the
search space simultaneously; NLME models must include either eta.cl (when mm = 0)
or eta.vmax (when mm = 1) to ensure at least one random effect on elimination.
For advanced model parameters not covered by \code{nlmixr2autoinit()},
initial estimates default to 1 before any transformation.
Users can provide custom initial estimates through the param_table argument.}
}

\strong{Custom Configuration Structure:}
When using search.space = "custom", the custom_config argument
must be provided as a list with four required elements:

\describe{
\item{route}{Character string: "bolus", "oral", or "mixed_iv_oral"}
\item{params}{Character vector of parameter names expected in string,
in the exact order they appear. Length of this vector must match length of
string.}
\item{param_dependencies}{Named list of functions where each function computes
a parameter value based on other parameters. Example: eta.vmax = function(mm) if (mm == 0) 0 else 1.
Use empty list if no dependencies exist.}
\item{fixed_params}{Named list of parameters with fixed values that are NOT
in string. These parameters are automatically passed to the model
generator. Use empty list if no fixed parameters exist.}
}

\strong{Using fixed_params:}
The fixed_params element specifies parameter values that remain constant
and do not appear in the string vector. This mechanism serves to:
\itemize{
\item Define model structure (e.g., compartment count, absorption type)
\item Fix certain parameters at specific values across all model runs
\item Keep the string vector shorter and focused on variable parameters
}

Caching System:
A two-level caching system avoids re-fitting identical models:
\itemize{
\item In-memory cache: Results stored in .modEnv during current session
\item File-based cache: Results loaded from CSV file specified by filename
}

To enable caching, set reuse_cache = 1 (default) and use consistent
filename across runs. Pass the same .modEnv object to subsequent
calls to maintain in-memory cache between model evaluations.
}
\examples{
\donttest{
# Example 1: IV model with pre-defined search space
param_table <- initialize_param_table()
param_table$init[param_table$Name == "lcl"] <- log(0.008)
param_table$init[param_table$Name == "lvc1cmpt"] <- log(0.6)
result <- mod.run(
  string = c(1, 0, 0, 0, 0, 0, 0, 0, 0, 1),
  dat = pheno_sd,
  search.space = "ivbase",
  param_table = param_table,
  saem.control = nlmixr2est::saemControl(logLik = TRUE,nBurn=200,nEm=300)
)

# Example 2: Oral model with pre-defined search space
param_table <- initialize_param_table()
param_table$init[param_table$Name == "lcl"] <- log(2.72)
param_table$init[param_table$Name == "lvc1cmpt"] <- log(31.5)
result <- mod.run(
  string = c(1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1),
  dat = theo_sd,
  search.space = "oralbase",
  param_table = param_table,
  saem.control = nlmixr2est::saemControl(logLik = TRUE,nBurn=200,nEm=300)
)

# Example 3: Simplified 1-compartment model with allometric scalling on
# Fix no.cmpt=1 and mcorr=0, vary only CL, Vc, and residual error
param_table <- initialize_param_table()
param_table$init[param_table$Name == "lcl"] <- log(0.008 *((70/3.5)^0.75))
param_table$init[param_table$Name == "lvc1cmpt"] <- log(0.6 *((70/3.5)))
simple_config <- list(
  route = "bolus",
  params = c("eta.vc", "mcorr", "rv"),
  param_dependencies = list(),
  fixed_params = list(
    no.cmpt = 1,
    eta.cl = 1,
    allometric_scaling = 1
  )
)
dat<-pheno_sd
dat$LOGWT<-log(dat$WT/70)
result <- mod.run(
  string = c(1, 1, 1),  # Only 3 values needed
  dat = dat,
  search.space = "custom",
  custom_config = simple_config,
  param_table = param_table,
  saem.control = nlmixr2est::saemControl(logLik = TRUE,nBurn=200,nEm=300)
)
}

}
\seealso{
\code{\link{spaceConfig}} for search space configuration details.
\code{\link{parseParams}} for parameter parsing.
\code{\link{ppkmodGen}} for model generation.
\code{\link{penaltyControl}} for penalty control settings.
}
\author{
Zhonghui Huang
}
