% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/mcc.R
\name{mcc}
\alias{mcc}
\title{Calculate Mean Cumulative Count (MCC)}
\usage{
mcc(
  data,
  id_var,
  time_var,
  cause_var,
  by = NULL,
  method = c("equation", "sci"),
  tstart_var = NULL,
  weights = NULL,
  adjust_times = TRUE,
  time_precision = 1e-06,
  include_details = TRUE
)
}
\arguments{
\item{data}{(\code{data.frame} or \code{tbl_df})\cr
A \code{data.frame} or tibble containing the required variables}

\item{id_var}{(\code{string})\cr
Name of the column containing participant identifiers}

\item{time_var}{(\code{string})\cr
Name of the column containing follow-up times}

\item{cause_var}{(\code{string})\cr
Name of the column containing event indicator values (1 = event of
interest, 2 = competing risk, 0 = censoring)}

\item{by}{(\code{string}, optional)\cr
Name of the column to group by for calculating MCC within subgroups.
If provided, MCC will be calculated separately for each level of this
variable}

\item{method}{(\code{string})\cr
Method to use for MCC calculation. Either \code{"equation"} (default) or
\code{"sci"} (sum of cumulative incidence)}

\item{tstart_var}{(\code{string})\cr
Name of the column containing start times of follow-up for incorporating
left truncation. Only allowed to be specified when \code{method = "sci"}. If
\code{NULL} (default), a constant value of \code{0} is used in calculation (i.e.,
right truncation only)}

\item{weights}{(\code{string}, optional)\cr
Name of the column containing weights for weighted MCC estimation.
Currently only supported with \code{method = "equation"}.
If provided, all weights must be non-negative and non-missing}

\item{adjust_times}{(\code{logical})\cr
If \code{TRUE} (default), automatically adjusts times to account for outcome
events and competing risk events occurring at the same time}

\item{time_precision}{(\code{numeric})\cr
Precision used for adjusting simultaneous events (default: 1e-6). Must
be a positive numeric value}

\item{include_details}{(\code{logical})\cr
Whether to include detailed calculation tables and intermediate objects
in the output. Default is \code{TRUE}, which returns all calculation details.
Setting to \code{FALSE} returns only the final MCC estimates, making the
function more efficient for bootstrapping}
}
\value{
An S3 object of class \code{"mcc"} with method-specific subclasses. The object contains:

When \code{include_details = TRUE} (default):

For \code{method = "equation"}:
\itemize{
\item \code{mcc_final}: A tibble with columns for \code{time} and \code{mcc}
\item \code{mcc_table}: A tibble with detailed calculation steps
\item \code{original_data}: The input \code{data} with standardized column names
\item \code{adjusted_data}: Present only if time adjustments were applied
}

For \code{method = "sci"}:
\itemize{
\item \code{mcc_final}: A tibble with columns for \code{time} and MCC (expressed as \code{SumCIs})
\item \code{sci_table}: A tibble with cumulative incidence for each event number and their sum
\item \code{all_cis}: A list of cumulative incidence data for each event number
\item \code{mcc_base}: A tibble with calculation details for the MCC
\item \code{original_data}: The input \code{data} with standardized column names
\item \code{adjusted_data}: Present only if time adjustments were applied
}

When \code{include_details = FALSE}:
\itemize{
\item \code{mcc_final}: A tibble with columns for \code{time} and \code{mcc} (or \code{SumCIs} for \code{method = "sci"})
}

All objects include metadata:
\itemize{
\item \code{method}: The method used for calculation
\item \code{weighted}: Logical indicating whether weighted estimation was used
\item \code{by_group}: Name of grouping variable (for grouped analyses)
\item \code{call}: The original function call
}

When \code{by} is specified, all tibbles contain an additional column with the
grouping variable values, and the object has the additional class \code{"mcc_grouped"}.
}
\description{
Calculates the mean cumulative count (MCC), which estimates the expected
cumulative number of events per person over time, while accounting for
potential competing risks and censoring. This function provides a unified
interface to two different estimation approaches: the Dong-Yasui
(\code{"equation"}) method and the sum of cumulative incidence (\code{"sci"}) method.

The \code{"equation"} method calculates MCC directly through probability
calculations, while the \code{"sci"} method derives MCC by summing the cumulative
incidence functions for each recurrent event. The two approaches yield
equivalent results in certain circumstances. When they do not, the choice
between methods depends on the specific outcome, analysis needs, and data
structure. See \code{vignette("choosing-between-methods")} for more details.
}
\examples{
# Attach dplyr
library(dplyr)
# Create sample data with recurrent events
df <- data.frame(
  id = c(1, 2, 3, 4, 4, 4, 4, 5, 5),
  time = c(8, 1, 5, 2, 6, 7, 8, 3, 3), # Times will be adjusted for id = 5
  cause = c(0, 0, 2, 1, 1, 1, 0, 1, 2)
 ) |>
  arrange(id, time)  # Sort the data by id and time

# Print the dataset
print("Hypothetical dataset from Dong et al. (2015):")
print(df)

# Calculate MCC using the equation method
mcc_eq <- mcc(df, id_var = "id", time_var = "time", cause_var = "cause")

# Print the S3 object
mcc_eq

# Get summary
summary(mcc_eq)

# Extract MCC estimates
mcc_estimates(mcc_eq)

# Extract calculation details
mcc_details(mcc_eq)

# Calculate MCC using the sum of cumulative incidence method
mcc_sci <- mcc(
  df,
  id_var = "id",
  time_var = "time",
  cause_var = "cause",
  method = "sci"
)

mcc_sci

# Clean up
rm(df, mcc_eq, mcc_sci)

}
\references{
Dong H, Robison LL, Leisenring WM, Martin LJ, Armstrong GT, Yasui Y.
Estimating the burden of recurrent events in the presence of competing
risks: the method of mean cumulative count. \emph{Am J Epidemiol}. 2015 Apr
1;181(7):532-40. doi: \href{https://www.doi.org/10.1093/aje/kwad031}{10.1093/aje/kwu289}

Gaber CE, Edwards JK, Lund JL, Peery AF, Richardson DB, Kinlaw AC.
Inverse Probability Weighting to Estimate Exposure Effects on the Burden of
Recurrent Outcomes in the Presence of Competing Events. \emph{Am J Epidemiol}.
2023;192(5):830-839. doi: \href{https://www.doi.org/10.1093/aje/kwad031}{10.1093/aje/kwad031}
}
