% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/knobi_proj.R
\name{knobi_proj}
\alias{knobi_proj}
\title{KBPM projections}
\arguments{
\item{knobi_results}{The output object of \code{\link{knobi_fit}} function (main package function).}

\item{env_results}{Optional. The output object of \code{\link{knobi_env}} function.}

\item{c}{Optional. A vector, data frame, or matrix specifying the catch values for each projected year. Multiple catch scenarios can be defined, with each column in the data frame or matrix representing a different scenario. For a single scenario, the length of the vector must match the number of projected years; for multiple scenarios, the number of rows must match the projected years. Projections are based on either future catch values or fishing mortality values, so only one of the two arguments, 'c' or 'f', is required.}

\item{f}{Optional. A vector, data frame, or matrix specifying the fishing mortality values for each projected year. Multiple fishing mortality scenarios can be defined, with each column in the data frame or matrix representing a different scenario. For a single scenario, the length of the vector must match the number of projected years; for multiple scenarios, the number of rows must match the projected years. Projections are based on either future catch values or fishing mortality values, so only one of the two arguments, 'c' or 'f', is required.}

\item{env}{Optional. Environmental variable(s) projections required if the environmental fit is provided to forecast the population and fishery dynamics. This fit considers the variable(s) selected in the \code{\link{knobi_env}} function. If the 'multicovar' argument of \code{\link{knobi_env}} is FALSE, this argument is a vector, data frame or matrix containing the values of the environmental covariates (unstandardized) (cols) for the projection years (rows). On the other hand, if the 'multicovar' argument of \code{\link{knobi_env}} is TRUE, the current argument must be a list, and each entry must be a data frame or matrix corresponding to each environmental scenario containing the values of the considered covariates for each scenario.}

\item{plot_out}{Logical. If TRUE, a file containing the plot of the retrospective fits is created. The default value is the input in the \code{\link{knobi_fit}} function.}

\item{plot_dir}{Optional. Directory to create the folder and save the plots. Required when 'plot_out=TRUE'. The default value is taken from the input of the \code{\link{knobi_fit}} function.}

\item{plot_filename}{Optional. Name of the folder that will contain the plots. Required when 'plot_out=TRUE'. The default value is taken from the input of the \code{\link{knobi_fit}} function.}
}
\value{
A list containing the projection results. \itemize{
\item df: data frame containing the projected time series. The rows correspond to the projection years, while the columns represent stock quantities: biomass or SSB, surplus production (SP), catch (C), and fishing mortality (F). Three additional columns specify the catch or fishing mortality scenario (Sc), the model used, and the environmental scenario (EnvSc).
\item plots: list containing the plots with the projections. Each plot is a panel plot that includes four subplots representing biomass or SSB, surplus production, fishing mortality, and catches for each catch or fishing mortality scenario and for each environmental scenario (if they is provided).}
The resulting plots are displayed in the plot window and are also saved (if plot_out="TRUE") in the  provided directory or in the same directory as \code{link{knobi_fit}}.
}
\description{
This function forecasts population and fishery dynamics under a defined set of management targets. It projects the time series of biomass (or SSB) and the surplus production for a set of future catch or fishing mortality scenarios.
}
\examples{

\donttest{

### Projecting through catch with no environmental information

# Then, create the data frame containing the selected catch for the projected
# years. In this illustration, within each scenario, the catch values are
# constant through the projected years. Three scenarios are considered:
# (i) catch value equal to the last historical catch multiplied by 1,
# (ii) last historical catch multiplied by 1.2 and
# (iii) last historical catch multiplied by 0.8.

catch<-rep(knobi_results$df$C[length(knobi_results$df$C)],5)

C<-data.frame(catch=catch,
              catch08=0.8*catch,
              catch12=1.2*catch)

# Then, knobi_proj function can be applied

knobi_proj(knobi_results, c=C)


### With environmental information

# In this case, in addition to the previous example, the 'knobi_env' example
# has to be run at first, where AMO variable was selected in the fit

# We include the future values of the environmental variable(s) in a data
# frame containing the environmental covariable values for the projected
# years. Three scenarios are considered:
# (i) Constant AMO equal to last year's AMO,
# (ii) Constant AMO equal to last year's AMO with a 50\% increment
# (iii) Constant AMO equal to last year's AMO with a 50\% decrease

last_AMO <- Env$AMO[length(Env$AMO)]
env <- data.frame( AMOi=rep(last_AMO,5),
                   AMOii=rep(last_AMO*1.5,5),
                   AMOiii=rep(last_AMO*0.5,5))

# Based on the previous objects we can apply the projection function.

knobi_proj(knobi_results, knobi_environmental, c=C, env=env)


### Through fishing mortality without environmental information

# Alternatively, projections can be based on fishing mortality.
# The scenarios presented below have been created from the estimated F_msy of
# knobi_fit analysis.

fmsy<-knobi_results$BRPs['F_MSY']
ff<-rep(fmsy,8)
f<-data.frame(f=ff,f12=ff*1.2,f08=ff*0.8)

knobi_proj(knobi_results, f=f)


### Through fishing mortality with environmental information

knobi_proj(knobi_results, f=f[1:5,], env_results=env_results, env=env)


# In case of multicovar<-TRUE in knobi_env, a list is required in which
# each item is a data frame for each environmental scenario

env<-list(climate_1=data.frame(AMO=c(0.2,0.2,0.3,0.3,0.4),
                              NAO=c(0.2,0.2,0.3,0.3,0.4)),
          climate_2=data.frame(AMO=c(0.2,0.3,0.4,0.5,0.6),
                              NAO=c(0.2,0.2,0.3,0.3,0.4)))

knobi_proj(knobi_results, knobi_environmental2, c=C, env=env)
}

}
\author{
\itemize{
\item{Anxo Paz}
\item{Marta Cousido-Rocha}
\item{Santiago Cerviño López}
\item{M. Grazia Pennino}
}
}
