/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5HFmodule.h" 

#include "H5private.h"   
#include "H5Eprivate.h"  
#include "H5FLprivate.h" 
#include "H5HFpkg.h"     
#include "H5MFprivate.h" 
#include "H5MMprivate.h" 
#include "H5VMprivate.h" 

#ifndef NDEBUG

#define H5HF_MAX_DIRECT_SIZE_LIMIT ((hsize_t)2 * 1024 * 1024 * 1024)

#define H5HF_WIDTH_LIMIT (64 * 1024)
#endif 

H5FL_DEFINE_STATIC(H5HF_hdr_t);

H5HF_hdr_t *
H5HF__hdr_alloc(H5F_t *f)
{
    H5HF_hdr_t *hdr       = NULL; 
    H5HF_hdr_t *ret_value = NULL; 

    FUNC_ENTER_PACKAGE

    
    assert(f);

    
    if (NULL == (hdr = H5FL_CALLOC(H5HF_hdr_t)))
        HGOTO_ERROR(H5E_HEAP, H5E_CANTALLOC, NULL, "allocation failed for fractal heap shared header");

    
    hdr->f           = f;
    hdr->sizeof_size = H5F_SIZEOF_SIZE(f);
    hdr->sizeof_addr = H5F_SIZEOF_ADDR(f);

    
    ret_value = hdr;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5HF__hdr_compute_free_space(H5HF_hdr_t *hdr, unsigned iblock_row)
{
    hsize_t  acc_heap_size;       
    hsize_t  iblock_size;         
    hsize_t  acc_dblock_free;     
    size_t   max_dblock_free;     
    unsigned curr_row;            
    herr_t   ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(hdr);
    assert(iblock_row >= hdr->man_dtable.max_direct_rows);

    
    acc_heap_size   = 0;
    acc_dblock_free = 0;
    max_dblock_free = 0;
    iblock_size     = hdr->man_dtable.row_block_size[iblock_row];
    curr_row        = 0;
    while (acc_heap_size < iblock_size) {
        acc_heap_size += hdr->man_dtable.row_block_size[curr_row] * hdr->man_dtable.cparam.width;
        acc_dblock_free += hdr->man_dtable.row_tot_dblock_free[curr_row] * hdr->man_dtable.cparam.width;
        if (hdr->man_dtable.row_max_dblock_free[curr_row] > max_dblock_free)
            max_dblock_free = hdr->man_dtable.row_max_dblock_free[curr_row];
        curr_row++;
    } 

    
    hdr->man_dtable.row_tot_dblock_free[iblock_row] = acc_dblock_free;
    hdr->man_dtable.row_max_dblock_free[iblock_row] = max_dblock_free;

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5HF__hdr_finish_init_phase1(H5HF_hdr_t *hdr)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(hdr);

    
    hdr->heap_off_size = (uint8_t)H5HF_SIZEOF_OFFSET_BITS(hdr->man_dtable.cparam.max_index);
    if (H5HF__dtable_init(&hdr->man_dtable) < 0)
        HGOTO_ERROR(H5E_HEAP, H5E_CANTINIT, FAIL, "can't initialize doubling table info");

    
    hdr->heap_len_size =
        (uint8_t)MIN(hdr->man_dtable.max_dir_blk_off_size, H5VM_limit_enc_size((uint64_t)hdr->max_man_size));

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5HF__hdr_finish_init_phase2(H5HF_hdr_t *hdr)
{
    unsigned u;                   
    herr_t   ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(hdr);

    
    for (u = 0; u < hdr->man_dtable.max_root_rows; u++) {
        if (u < hdr->man_dtable.max_direct_rows) {
            hdr->man_dtable.row_tot_dblock_free[u] =
                hdr->man_dtable.row_block_size[u] - H5HF_MAN_ABS_DIRECT_OVERHEAD(hdr);
            H5_CHECKED_ASSIGN(hdr->man_dtable.row_max_dblock_free[u], size_t,
                              hdr->man_dtable.row_tot_dblock_free[u], hsize_t);
        } 
        else if (H5HF__hdr_compute_free_space(hdr, u) < 0)
            HGOTO_ERROR(H5E_HEAP, H5E_CANTINIT, FAIL,
                        "can't initialize direct block free space for indirect block");
    } 

    
    if (H5HF__man_iter_init(&hdr->next_block) < 0)
        HGOTO_ERROR(H5E_HEAP, H5E_CANTINIT, FAIL, "can't initialize space search block iterator");

    
    if (H5HF__huge_init(hdr) < 0)
        HGOTO_ERROR(H5E_HEAP, H5E_CANTINIT, FAIL, "can't initialize info for tracking huge objects");

    
    if (H5HF__tiny_init(hdr) < 0)
        HGOTO_ERROR(H5E_HEAP, H5E_CANTINIT, FAIL, "can't initialize info for tracking tiny objects");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5HF__hdr_finish_init(H5HF_hdr_t *hdr)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(hdr);

    
    if (H5HF__hdr_finish_init_phase1(hdr) < 0)
        HGOTO_ERROR(H5E_HEAP, H5E_CANTINIT, FAIL, "can't finish phase #1 of header final initialization");

    
    if (H5HF__hdr_finish_init_phase2(hdr) < 0)
        HGOTO_ERROR(H5E_HEAP, H5E_CANTINIT, FAIL, "can't finish phase #2 of header final initialization");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

haddr_t
H5HF__hdr_create(H5F_t *f, const H5HF_create_t *cparam)
{
    H5HF_hdr_t *hdr = NULL;              
    size_t      dblock_overhead;         
    haddr_t     ret_value = HADDR_UNDEF; 

    FUNC_ENTER_PACKAGE

    
    assert(f);
    assert(cparam);

#ifndef NDEBUG
    
    if (cparam->managed.width == 0)
        HGOTO_ERROR(H5E_HEAP, H5E_BADVALUE, HADDR_UNDEF, "width must be greater than zero");
    if (cparam->managed.width > H5HF_WIDTH_LIMIT)
        HGOTO_ERROR(H5E_HEAP, H5E_BADVALUE, HADDR_UNDEF, "width too large");
    if (!POWER_OF_TWO(cparam->managed.width))
        HGOTO_ERROR(H5E_HEAP, H5E_BADVALUE, HADDR_UNDEF, "width not power of two");
    if (cparam->managed.start_block_size == 0)
        HGOTO_ERROR(H5E_HEAP, H5E_BADVALUE, HADDR_UNDEF, "starting block size must be greater than zero");
    if (!POWER_OF_TWO(cparam->managed.start_block_size))
        HGOTO_ERROR(H5E_HEAP, H5E_BADVALUE, HADDR_UNDEF, "starting block size not power of two");
    if (cparam->managed.max_direct_size == 0)
        HGOTO_ERROR(H5E_HEAP, H5E_BADVALUE, HADDR_UNDEF, "max. direct block size must be greater than zero");
    if (cparam->managed.max_direct_size > H5HF_MAX_DIRECT_SIZE_LIMIT)
        HGOTO_ERROR(H5E_HEAP, H5E_BADVALUE, HADDR_UNDEF, "max. direct block size too large");
    if (!POWER_OF_TWO(cparam->managed.max_direct_size))
        HGOTO_ERROR(H5E_HEAP, H5E_BADVALUE, HADDR_UNDEF, "max. direct block size not power of two");
    if (cparam->managed.max_direct_size < cparam->max_man_size)
        HGOTO_ERROR(H5E_HEAP, H5E_BADVALUE, HADDR_UNDEF,
                    "max. direct block size not large enough to hold all managed blocks");
    if (cparam->managed.max_index == 0)
        HGOTO_ERROR(H5E_HEAP, H5E_BADVALUE, HADDR_UNDEF, "max. heap size must be greater than zero");
#endif 

    
    if (NULL == (hdr = H5HF__hdr_alloc(f)))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, HADDR_UNDEF, "can't allocate space for shared heap info");

#ifndef NDEBUG
    if (cparam->managed.max_index > (unsigned)(8 * hdr->sizeof_size))
        HGOTO_ERROR(H5E_HEAP, H5E_BADVALUE, HADDR_UNDEF, "max. heap size too large for file");
#endif 

    
    hdr->max_man_size     = cparam->max_man_size;
    hdr->checksum_dblocks = cparam->checksum_dblocks;
    H5MM_memcpy(&(hdr->man_dtable.cparam), &(cparam->managed), sizeof(H5HF_dtable_cparam_t));

    
    hdr->man_dtable.table_addr = HADDR_UNDEF;

    
    hdr->fs_addr = HADDR_UNDEF;

    
    hdr->huge_bt2_addr = HADDR_UNDEF;

    
    
    if (H5HF__hdr_finish_init_phase1(hdr) < 0)
        HGOTO_ERROR(H5E_HEAP, H5E_CANTINIT, HADDR_UNDEF,
                    "can't finish phase #1 of header final initialization");

    
    
    if (cparam->pline.nused > 0) {
        
        if (H5Z_can_apply_direct(&(cparam->pline)) < 0)
            HGOTO_ERROR(H5E_HEAP, H5E_CANTINIT, HADDR_UNDEF, "I/O filters can't operate on this heap");

        
        hdr->checked_filters = true;

        
        if (H5Z_set_local_direct(&(cparam->pline)) < 0)
            HGOTO_ERROR(H5E_HEAP, H5E_CANTINIT, HADDR_UNDEF, "unable to set local filter parameters");

        
        if (NULL == H5O_msg_copy(H5O_PLINE_ID, &(cparam->pline), &(hdr->pline)))
            HGOTO_ERROR(H5E_HEAP, H5E_CANTCOPY, HADDR_UNDEF, "can't copy I/O filter pipeline");

        
        if (H5O_pline_set_version(hdr->f, &(hdr->pline)) < 0)
            HGOTO_ERROR(H5E_HEAP, H5E_CANTSET, HADDR_UNDEF, "can't set version of I/O filter pipeline");

        
        if (0 == (hdr->filter_len = (unsigned)H5O_msg_raw_size(hdr->f, H5O_PLINE_ID, false, &(hdr->pline))))
            HGOTO_ERROR(H5E_HEAP, H5E_CANTGETSIZE, HADDR_UNDEF, "can't get I/O filter pipeline size");

        
        hdr->heap_size = H5HF_HEADER_SIZE(hdr) 
                         + hdr->sizeof_size    
                         + 4                   
                         + hdr->filter_len;    
    }                                          
    else {
        
        hdr->heap_size = H5HF_HEADER_SIZE(hdr);

        
        hdr->checked_filters = true;
    } 

    
    
    switch (cparam->id_len) {
        case 0: 
            hdr->id_len = (unsigned)1 + hdr->heap_off_size + hdr->heap_len_size;
            break;

        case 1: 
            if (hdr->filter_len > 0)
                hdr->id_len = (unsigned)1         
                              + hdr->sizeof_addr  
                              + hdr->sizeof_size  
                              + 4                 
                              + hdr->sizeof_size; 
            else
                hdr->id_len = (unsigned)1         
                              + hdr->sizeof_addr  
                              + hdr->sizeof_size; 
            break;

        default: 
            
            if (cparam->id_len < (1 + hdr->heap_off_size + hdr->heap_len_size))
                HGOTO_ERROR(H5E_HEAP, H5E_BADRANGE, HADDR_UNDEF,
                            "ID length not large enough to hold object IDs");
            else if (cparam->id_len > H5HF_MAX_ID_LEN)
                HGOTO_ERROR(H5E_HEAP, H5E_BADRANGE, HADDR_UNDEF,
                            "ID length too large to store tiny object lengths");

            
            hdr->id_len = cparam->id_len;
            break;
    } 

    
    
    if (H5HF__hdr_finish_init_phase2(hdr) < 0)
        HGOTO_ERROR(H5E_HEAP, H5E_CANTINIT, HADDR_UNDEF,
                    "can't finish phase #2 of header final initialization");

    
    dblock_overhead = H5HF_MAN_ABS_DIRECT_OVERHEAD(hdr);
    if ((cparam->managed.max_direct_size - dblock_overhead) < cparam->max_man_size)
        HGOTO_ERROR(H5E_HEAP, H5E_BADVALUE, HADDR_UNDEF,
                    "max. direct block size not large enough to hold all managed blocks");

    
    if (HADDR_UNDEF == (hdr->heap_addr = H5MF_alloc(f, H5FD_MEM_FHEAP_HDR, (hsize_t)hdr->heap_size)))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, HADDR_UNDEF, "file allocation failed for fractal heap header");

    
    if (H5AC_insert_entry(f, H5AC_FHEAP_HDR, hdr->heap_addr, hdr, H5AC__NO_FLAGS_SET) < 0)
        HGOTO_ERROR(H5E_HEAP, H5E_CANTINSERT, HADDR_UNDEF, "can't add fractal heap header to cache");

    
    ret_value = hdr->heap_addr;

done:
    if (!H5_addr_defined(ret_value) && hdr)
        if (H5HF__hdr_free(hdr) < 0)
            HDONE_ERROR(H5E_HEAP, H5E_CANTRELEASE, HADDR_UNDEF, "unable to release fractal heap header");

    FUNC_LEAVE_NOAPI(ret_value)
} 

H5HF_hdr_t *
H5HF__hdr_protect(H5F_t *f, haddr_t addr, unsigned flags)
{
    H5HF_hdr_cache_ud_t cache_udata;      
    H5HF_hdr_t         *hdr;              
    H5HF_hdr_t         *ret_value = NULL; 

    FUNC_ENTER_PACKAGE

    
    assert(f);
    assert(H5_addr_defined(addr));

    
    assert((flags & (unsigned)(~H5AC__READ_ONLY_FLAG)) == 0);

    
    cache_udata.f = f;

    
    if (NULL == (hdr = (H5HF_hdr_t *)H5AC_protect(f, H5AC_FHEAP_HDR, addr, &cache_udata, flags)))
        HGOTO_ERROR(H5E_HEAP, H5E_CANTPROTECT, NULL, "unable to protect fractal heap header");

    
    hdr->heap_addr = addr;

    
    hdr->f = f;

    
    ret_value = hdr;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5HF__hdr_incr(H5HF_hdr_t *hdr)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(hdr);

    
    if (hdr->rc == 0)
        if (H5AC_pin_protected_entry(hdr) < 0)
            HGOTO_ERROR(H5E_HEAP, H5E_CANTPIN, FAIL, "unable to pin fractal heap header");

    
    hdr->rc++;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5HF__hdr_decr(H5HF_hdr_t *hdr)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(hdr);
    assert(hdr->rc);

    
    hdr->rc--;

    
    if (hdr->rc == 0) {
        assert(hdr->file_rc == 0);
        if (H5AC_unpin_entry(hdr) < 0)
            HGOTO_ERROR(H5E_HEAP, H5E_CANTUNPIN, FAIL, "unable to unpin fractal heap header");
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5HF__hdr_fuse_incr(H5HF_hdr_t *hdr)
{
    FUNC_ENTER_PACKAGE_NOERR

    
    assert(hdr);

    
    hdr->file_rc++;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

size_t
H5HF__hdr_fuse_decr(H5HF_hdr_t *hdr)
{
    FUNC_ENTER_PACKAGE_NOERR

    
    assert(hdr);
    assert(hdr->file_rc);

    
    hdr->file_rc--;

    FUNC_LEAVE_NOAPI(hdr->file_rc)
} 

herr_t
H5HF__hdr_dirty(H5HF_hdr_t *hdr)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(hdr);

    
    if (hdr->filter_len > 0)
        if (H5AC_resize_entry(hdr, (size_t)hdr->heap_size) < 0)
            HGOTO_ERROR(H5E_HEAP, H5E_CANTRESIZE, FAIL, "unable to resize fractal heap header");

    
    if (H5AC_mark_entry_dirty(hdr) < 0)
        HGOTO_ERROR(H5E_HEAP, H5E_CANTMARKDIRTY, FAIL, "unable to mark fractal heap header as dirty");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5HF__hdr_adj_free(H5HF_hdr_t *hdr, ssize_t amt)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(hdr);

    
    assert(amt > 0 || hdr->total_man_free >= (hsize_t)-amt);
    hdr->total_man_free = (hsize_t)((hssize_t)hdr->total_man_free + amt);

    
    if (H5HF__hdr_dirty(hdr) < 0)
        HGOTO_ERROR(H5E_HEAP, H5E_CANTDIRTY, FAIL, "can't mark heap header as dirty");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5HF__hdr_adjust_heap(H5HF_hdr_t *hdr, hsize_t new_size, hssize_t extra_free)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(hdr);

    
    hdr->man_size = new_size;

    
    assert(extra_free > 0 || hdr->total_man_free >= (hsize_t)-extra_free);
    hdr->total_man_free = (hsize_t)((hssize_t)hdr->total_man_free + extra_free);

    
    if (H5HF__hdr_dirty(hdr) < 0)
        HGOTO_ERROR(H5E_HEAP, H5E_CANTDIRTY, FAIL, "can't mark header as dirty");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5HF__hdr_inc_alloc(H5HF_hdr_t *hdr, size_t alloc_size)
{
    FUNC_ENTER_PACKAGE_NOERR

    
    assert(hdr);
    assert(alloc_size);

    
    hdr->man_alloc_size += alloc_size;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

herr_t
H5HF__hdr_start_iter(H5HF_hdr_t *hdr, H5HF_indirect_t *iblock, hsize_t curr_off, unsigned curr_entry)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(hdr);
    assert(iblock);

    
    if (H5HF__man_iter_start_entry(hdr, &hdr->next_block, iblock, curr_entry) < 0)
        HGOTO_ERROR(H5E_HEAP, H5E_CANTINIT, FAIL, "can't initialize block iterator");

    
    hdr->man_iter_off = curr_off;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5HF__hdr_reset_iter(H5HF_hdr_t *hdr, hsize_t curr_off)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(hdr);

    
    if (H5HF__man_iter_reset(&hdr->next_block) < 0)
        HGOTO_ERROR(H5E_HEAP, H5E_CANTRELEASE, FAIL, "can't reset block iterator");

    
    hdr->man_iter_off = curr_off;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5HF__hdr_skip_blocks(H5HF_hdr_t *hdr, H5HF_indirect_t *iblock, unsigned start_entry, unsigned nentries)
{
    unsigned row, col;            
    hsize_t  sect_size;           
    herr_t   ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(hdr);
    assert(iblock);
    assert(nentries);

    
    row       = start_entry / hdr->man_dtable.cparam.width;
    col       = start_entry % hdr->man_dtable.cparam.width;
    sect_size = H5HF__dtable_span_size(&hdr->man_dtable, row, col, nentries);
    assert(sect_size > 0);

    
    if (H5HF__hdr_inc_iter(hdr, sect_size, nentries) < 0)
        HGOTO_ERROR(H5E_HEAP, H5E_CANTRELEASE, FAIL, "can't increase allocated heap size");

    
    if (H5HF__sect_indirect_add(hdr, iblock, start_entry, nentries) < 0)
        HGOTO_ERROR(H5E_HEAP, H5E_CANTINIT, FAIL,
                    "can't create indirect section for indirect block's free space");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5HF__hdr_update_iter(H5HF_hdr_t *hdr, size_t min_dblock_size)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(hdr);
    assert(min_dblock_size > 0);

    
    if (hdr->man_dtable.curr_root_rows == 0) {
        if (H5HF__man_iblock_root_create(hdr, min_dblock_size) < 0)
            HGOTO_ERROR(H5E_HEAP, H5E_CANTEXTEND, FAIL, "unable to create root indirect block");
    } 
    else {
        H5HF_indirect_t *iblock;                 
        bool             walked_up, walked_down; 
        unsigned         next_row;               
        unsigned         next_entry;             
        unsigned         min_dblock_row;         

        
        min_dblock_row = H5HF__dtable_size_to_row(&hdr->man_dtable, min_dblock_size);

        
        if (!H5HF__man_iter_ready(&hdr->next_block)) {
            
            if (H5HF__man_iter_start_offset(hdr, &hdr->next_block, hdr->man_iter_off) < 0)
                HGOTO_ERROR(H5E_HEAP, H5E_CANTINIT, FAIL, "unable to set block iterator location");
        } 

        
        if (H5HF__man_iter_curr(&hdr->next_block, &next_row, NULL, &next_entry, &iblock) < 0)
            HGOTO_ERROR(H5E_HEAP, H5E_CANTGET, FAIL, "unable to retrieve current block iterator location");

        
        if (min_dblock_row > next_row && next_row < iblock->nrows) {
            unsigned min_entry;    
            unsigned skip_entries; 

            
            min_entry = min_dblock_row * hdr->man_dtable.cparam.width;
            if (min_dblock_row >= iblock->nrows)
                skip_entries = (iblock->nrows * hdr->man_dtable.cparam.width) - next_entry;
            else
                skip_entries = min_entry - next_entry;

            
            if (H5HF__hdr_skip_blocks(hdr, iblock, next_entry, skip_entries) < 0)
                HGOTO_ERROR(H5E_HEAP, H5E_CANTDEC, FAIL, "can't add skipped blocks to heap's free space");

            
            if (H5HF__man_iter_curr(&hdr->next_block, &next_row, NULL, &next_entry, &iblock) < 0)
                HGOTO_ERROR(H5E_HEAP, H5E_CANTGET, FAIL,
                            "unable to retrieve current block iterator location");
        } 

        do {
            
            walked_up = walked_down = false;

            
            
            while (next_row >= iblock->nrows) {
                
                if (iblock->parent == NULL) {
                    if (H5HF__man_iblock_root_double(hdr, min_dblock_size) < 0)
                        HGOTO_ERROR(H5E_HEAP, H5E_CANTEXTEND, FAIL, "unable to double root indirect block");
                } 
                else {
                    
                    if (H5HF__man_iter_up(&hdr->next_block) < 0)
                        HGOTO_ERROR(H5E_HEAP, H5E_CANTNEXT, FAIL,
                                    "unable to advance current block iterator location");

                    
                    if (H5HF__man_iter_next(hdr, &hdr->next_block, 1) < 0)
                        HGOTO_ERROR(H5E_HEAP, H5E_CANTINC, FAIL, "can't advance fractal heap block location");
                } 

                
                if (H5HF__man_iter_curr(&hdr->next_block, &next_row, NULL, &next_entry, &iblock) < 0)
                    HGOTO_ERROR(H5E_HEAP, H5E_CANTGET, FAIL,
                                "unable to retrieve current block iterator location");

                
                walked_up = true;
            } 

            
            
            if (next_row >= hdr->man_dtable.max_direct_rows) {
                unsigned child_nrows; 

                assert(!H5_addr_defined(iblock->ents[next_entry].addr));

                
                child_nrows =
                    H5HF__dtable_size_to_rows(&hdr->man_dtable, hdr->man_dtable.row_block_size[next_row]);

                
                
                if (hdr->man_dtable.row_block_size[child_nrows - 1] < min_dblock_size) {
                    unsigned child_rows_needed; 
                    unsigned child_entry;       

                    
                    child_rows_needed = (H5VM_log2_of2((uint32_t)min_dblock_size) -
                                         H5VM_log2_of2((uint32_t)hdr->man_dtable.cparam.start_block_size)) +
                                        2;
                    assert(child_rows_needed > child_nrows);
                    child_entry =
                        (next_row + (child_rows_needed - child_nrows)) * hdr->man_dtable.cparam.width;
                    if (child_entry > (iblock->nrows * hdr->man_dtable.cparam.width))
                        child_entry = iblock->nrows * hdr->man_dtable.cparam.width;

                    
                    if (H5HF__hdr_skip_blocks(hdr, iblock, next_entry, (child_entry - next_entry)) < 0)
                        HGOTO_ERROR(H5E_HEAP, H5E_CANTDEC, FAIL,
                                    "can't add skipped blocks to heap's free space");
                } 
                else {
                    H5HF_indirect_t *new_iblock;      
                    bool             did_protect;     
                    haddr_t          new_iblock_addr; 

                    
                    if (H5HF__man_iblock_create(hdr, iblock, next_entry, child_nrows, child_nrows,
                                                &new_iblock_addr) < 0)
                        HGOTO_ERROR(H5E_HEAP, H5E_CANTALLOC, FAIL,
                                    "can't allocate fractal heap indirect block");

                    
                    if (NULL == (new_iblock = H5HF__man_iblock_protect(hdr, new_iblock_addr, child_nrows,
                                                                       iblock, next_entry, false,
                                                                       H5AC__NO_FLAGS_SET, &did_protect)))
                        HGOTO_ERROR(H5E_HEAP, H5E_CANTPROTECT, FAIL,
                                    "unable to protect fractal heap indirect block");

                    
                    if (H5HF__man_iter_down(&hdr->next_block, new_iblock) < 0)
                        HGOTO_ERROR(H5E_HEAP, H5E_CANTNEXT, FAIL,
                                    "unable to advance current block iterator location");

                    
                    if (min_dblock_size > hdr->man_dtable.cparam.start_block_size) {
                        unsigned new_entry; 

                        
                        new_entry = hdr->man_dtable.cparam.width * min_dblock_row;

                        
                        if (H5HF__hdr_skip_blocks(hdr, new_iblock, 0, new_entry) < 0)
                            HGOTO_ERROR(H5E_HEAP, H5E_CANTDEC, FAIL,
                                        "can't add skipped blocks to heap's free space");
                    } 

                    
                    if (H5HF__man_iblock_unprotect(new_iblock, H5AC__NO_FLAGS_SET, did_protect) < 0)
                        HGOTO_ERROR(H5E_HEAP, H5E_CANTUNPROTECT, FAIL,
                                    "unable to release fractal heap indirect block");
                } 

                
                if (H5HF__man_iter_curr(&hdr->next_block, &next_row, NULL, &next_entry, &iblock) < 0)
                    HGOTO_ERROR(H5E_HEAP, H5E_CANTGET, FAIL,
                                "unable to retrieve current block iterator location");

                
                walked_down = true;
            } 
        } while (walked_down || walked_up);
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5HF__hdr_inc_iter(H5HF_hdr_t *hdr, hsize_t adv_size, unsigned nentries)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(hdr);
    assert(nentries);

    
    if (hdr->next_block.curr)
        if (H5HF__man_iter_next(hdr, &hdr->next_block, nentries) < 0)
            HGOTO_ERROR(H5E_HEAP, H5E_CANTNEXT, FAIL, "unable to advance current block iterator location");

    
    hdr->man_iter_off += adv_size;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5HF__hdr_reverse_iter(H5HF_hdr_t *hdr, haddr_t dblock_addr)
{
    H5HF_indirect_t *iblock;              
    unsigned         curr_entry;          
    bool             walked_down;         
    bool             walked_up;           
    herr_t           ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(hdr);

    
    if (!H5HF__man_iter_ready(&hdr->next_block))
        
        if (H5HF__man_iter_start_offset(hdr, &hdr->next_block, hdr->man_iter_off) < 0)
            HGOTO_ERROR(H5E_HEAP, H5E_CANTINIT, FAIL, "unable to set block iterator location");

    

    
    if (H5HF__man_iter_curr(&hdr->next_block, NULL, NULL, &curr_entry, &iblock) < 0)
        HGOTO_ERROR(H5E_HEAP, H5E_CANTGET, FAIL, "unable to retrieve current block iterator information");

    
    curr_entry--;

    
    do {
        int tmp_entry; 

        
        walked_down = false;
        walked_up   = false;

        
        
        tmp_entry = (int)curr_entry;
        while (tmp_entry >= 0 && (H5_addr_eq(iblock->ents[tmp_entry].addr, dblock_addr) ||
                                  !H5_addr_defined(iblock->ents[tmp_entry].addr)))
            tmp_entry--;
        
        if (tmp_entry < 0) {
            
            if (iblock->parent) {
                
                if (H5HF__man_iter_up(&hdr->next_block) < 0)
                    HGOTO_ERROR(H5E_HEAP, H5E_CANTNEXT, FAIL,
                                "unable to move current block iterator location up");

                
                if (H5HF__man_iter_curr(&hdr->next_block, NULL, NULL, &curr_entry, &iblock) < 0)
                    HGOTO_ERROR(H5E_HEAP, H5E_CANTGET, FAIL,
                                "unable to retrieve current block iterator information");

                
                curr_entry--;

                
                walked_up = true;
            } 
            else {
                
                hdr->man_iter_off = 0;

                
                if (H5HF__man_iter_reset(&hdr->next_block) < 0)
                    HGOTO_ERROR(H5E_HEAP, H5E_CANTRELEASE, FAIL, "can't reset block iterator");
            } 
        }     
        else {
            unsigned row; 

            curr_entry = (unsigned)tmp_entry;

            
            row = curr_entry / hdr->man_dtable.cparam.width;
            if (row < hdr->man_dtable.max_direct_rows) {
                
                curr_entry++;

                
                if (H5HF__man_iter_set_entry(hdr, &hdr->next_block, curr_entry) < 0)
                    HGOTO_ERROR(H5E_HEAP, H5E_CANTSET, FAIL, "unable to set current block iterator location");

                
                hdr->man_iter_off = iblock->block_off;
                hdr->man_iter_off += hdr->man_dtable.row_block_off[curr_entry / hdr->man_dtable.cparam.width];
                hdr->man_iter_off +=
                    hdr->man_dtable.row_block_size[curr_entry / hdr->man_dtable.cparam.width] *
                    (curr_entry % hdr->man_dtable.cparam.width);
            } 
            else {
                H5HF_indirect_t *child_iblock; 
                bool             did_protect;  
                unsigned         child_nrows;  

                
                child_nrows =
                    H5HF__dtable_size_to_rows(&hdr->man_dtable, hdr->man_dtable.row_block_size[row]);

                
                if (NULL == (child_iblock = H5HF__man_iblock_protect(hdr, iblock->ents[curr_entry].addr,
                                                                     child_nrows, iblock, curr_entry, false,
                                                                     H5AC__NO_FLAGS_SET, &did_protect)))
                    HGOTO_ERROR(H5E_HEAP, H5E_CANTPROTECT, FAIL,
                                "unable to protect fractal heap indirect block");

                
                if (H5HF__man_iter_set_entry(hdr, &hdr->next_block, curr_entry) < 0)
                    HGOTO_ERROR(H5E_HEAP, H5E_CANTSET, FAIL, "unable to set current block iterator location");

                
                if (H5HF__man_iter_down(&hdr->next_block, child_iblock) < 0)
                    HGOTO_ERROR(H5E_HEAP, H5E_CANTNEXT, FAIL,
                                "unable to advance current block iterator location");

                
                iblock     = child_iblock;
                curr_entry = (child_iblock->nrows * hdr->man_dtable.cparam.width) - 1;

                
                if (H5HF__man_iblock_unprotect(child_iblock, H5AC__NO_FLAGS_SET, did_protect) < 0)
                    HGOTO_ERROR(H5E_HEAP, H5E_CANTUNPROTECT, FAIL,
                                "unable to release fractal heap indirect block");

                
                walked_down = true;
            } 
        }     
    } while (walked_down || walked_up);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5HF__hdr_empty(H5HF_hdr_t *hdr)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(hdr);

    
    if (H5HF__man_iter_ready(&hdr->next_block))
        if (H5HF__man_iter_reset(&hdr->next_block) < 0)
            HGOTO_ERROR(H5E_HEAP, H5E_CANTRELEASE, FAIL, "can't reset block iterator");

    
    hdr->man_size       = 0;
    hdr->man_alloc_size = 0;

    
    hdr->man_dtable.curr_root_rows = 0;
    hdr->man_dtable.table_addr     = HADDR_UNDEF;

    
    hdr->man_iter_off = 0;

    
    hdr->total_man_free = 0;

    
    if (H5HF__hdr_dirty(hdr) < 0)
        HGOTO_ERROR(H5E_HEAP, H5E_CANTDIRTY, FAIL, "can't mark header as dirty");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5HF__hdr_free(H5HF_hdr_t *hdr)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(hdr);

    
    if (H5HF__dtable_dest(&hdr->man_dtable) < 0)
        HGOTO_ERROR(H5E_HEAP, H5E_CANTFREE, FAIL, "unable to destroy fractal heap doubling table");

    
    if (hdr->pline.nused)
        if (H5O_msg_reset(H5O_PLINE_ID, &(hdr->pline)) < 0)
            HGOTO_ERROR(H5E_HEAP, H5E_CANTFREE, FAIL, "unable to reset I/O pipeline message");

    
    hdr = H5FL_FREE(H5HF_hdr_t, hdr);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5HF__hdr_delete(H5HF_hdr_t *hdr)
{
    unsigned cache_flags = H5AC__NO_FLAGS_SET; 
    herr_t   ret_value   = SUCCEED;            

    FUNC_ENTER_PACKAGE

    
    assert(hdr);
    assert(!hdr->file_rc);

#ifndef NDEBUG
    {
        unsigned hdr_status = 0; 

        
        if (H5AC_get_entry_status(hdr->f, hdr->heap_addr, &hdr_status) < 0)
            HGOTO_ERROR(H5E_HEAP, H5E_CANTGET, FAIL, "unable to check metadata cache status for heap header");

        
        assert(hdr_status & H5AC_ES__IN_CACHE);
        assert(hdr_status & H5AC_ES__IS_PROTECTED);
    }  
#endif 

    
    
    if (H5_addr_defined(hdr->fs_addr))
        
        if (H5HF__space_delete(hdr) < 0)
            HGOTO_ERROR(H5E_HEAP, H5E_CANTFREE, FAIL, "unable to release fractal heap free space manager");

    
    if (H5_addr_defined(hdr->man_dtable.table_addr)) {
        if (hdr->man_dtable.curr_root_rows == 0) {
            hsize_t dblock_size; 

            
            if (hdr->filter_len > 0) {
                
                dblock_size = hdr->pline_root_direct_size;

                
                hdr->pline_root_direct_size        = 0;
                hdr->pline_root_direct_filter_mask = 0;
            } 
            else
                dblock_size = hdr->man_dtable.cparam.start_block_size;

            
            if (H5HF__man_dblock_delete(hdr->f, hdr->man_dtable.table_addr, dblock_size) < 0)
                HGOTO_ERROR(H5E_HEAP, H5E_CANTFREE, FAIL, "unable to release fractal heap root direct block");
        } 
        else {
            
            if (H5HF__man_iblock_delete(hdr, hdr->man_dtable.table_addr, hdr->man_dtable.curr_root_rows, NULL,
                                        0) < 0)
                HGOTO_ERROR(H5E_HEAP, H5E_CANTFREE, FAIL,
                            "unable to release fractal heap root indirect block");
        } 
    }     

    
    if (H5_addr_defined(hdr->huge_bt2_addr)) {
        
        if (H5HF__huge_delete(hdr) < 0)
            HGOTO_ERROR(H5E_HEAP, H5E_CANTFREE, FAIL,
                        "unable to release fractal heap 'huge' objects and tracker");
    } 

    
    cache_flags |= H5AC__DIRTIED_FLAG | H5AC__DELETED_FLAG | H5AC__FREE_FILE_SPACE_FLAG;

done:
    
    if (H5AC_unprotect(hdr->f, H5AC_FHEAP_HDR, hdr->heap_addr, hdr, cache_flags) < 0)
        HDONE_ERROR(H5E_HEAP, H5E_CANTUNPROTECT, FAIL, "unable to release fractal heap header");

    FUNC_LEAVE_NOAPI(ret_value)
} 
