% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/utils-common.R
\name{gs.build_proc_grps}
\alias{gs.build_proc_grps}
\title{Build reconciliation processing groups}
\usage{
gs.build_proc_grps(
  ts_yr_vec,
  ts_per_vec,
  n_per,
  ts_freq,
  temporal_grp_periodicity,
  temporal_grp_start
)
}
\arguments{
\item{ts_yr_vec}{(mandatory)

Vector of the time series year (time unit) values (see \code{\link[=gs.time2year]{gs.time2year()}}).}

\item{ts_per_vec}{(mandatory)

Vector of the time series period (cycle) values (see \code{\link[=gs.time2per]{gs.time2per()}}).}

\item{n_per}{(mandatory)

Time series length (number of periods).}

\item{ts_freq}{(mandatory)

Time series frequency (see \code{\link[stats:time]{stats::frequency()}}).}

\item{temporal_grp_periodicity}{(mandatory)

Number of periods in temporal groups.}

\item{temporal_grp_start}{(mandatory)

First period of temporal groups.}
}
\value{
A data frame with the following variables (columns):
\itemize{
\item \code{grp}         : integer vector identifying the processing group (\verb{1:<number-of-groups>}).
\item \code{beg_per}     : integer vector identifying the first period of the processing group.
\item \code{end_per}     : integer vector identifying the last period of the processing group.
\item \code{complete_grp}: logical vector indicating if the processing group corresponds to a complete temporal group.
}
}
\description{
\if{html,text}{(\emph{version française: 
\url{https://StatCan.github.io/gensol-gseries/fr/reference/gs.build_proc_grps.html}})}

This function builds the processing groups data frame for reconciliation problems. It is used internally by
\code{\link[=tsraking_driver]{tsraking_driver()}} and \code{\link[=tsbalancing]{tsbalancing()}}.
}
\section{Processing groups}{
The set of periods of a given reconciliation (raking or balancing) problem is called a \emph{processing group}
and either corresponds to:
\itemize{
\item a \strong{single period} with period-by-period processing or, when preserving temporal totals, for the individual
periods of an incomplete temporal group (e.g., an incomplete year)
\item or the \strong{set of periods of a complete temporal group} (e.g., a complete year) when preserving temporal
totals.
}

The total number of processing groups (total number of reconciliation problems) depends on the set of
periods in the input time series object (argument \code{in_ts}) and on the value of arguments
\code{temporal_grp_periodicity} and \code{temporal_grp_start}.

Common scenarios include \code{temporal_grp_periodicity = 1} (default) for period-by period processing without
temporal total preservation and \code{temporal_grp_periodicity = frequency(in_ts)} for the preservation of annual
totals (calendar years by default). Argument \code{temporal_grp_start} allows the specification of other types of
(\emph{non-calendar}) years. E.g., fiscal years starting on April correspond to \code{temporal_grp_start = 4} with monthly
data and \code{temporal_grp_start = 2} with quarterly data. Preserving quarterly totals with monthly data would
correspond to \ifelse{latex}{\code{temporal_grp _periodicity = 3}}{\code{temporal_grp_periodicity = 3}}.

By default, temporal groups covering more than a year (i.e., corresponding to \ifelse{latex}{\code{
temporal_grp _periodicity > frequency(in_ts)}}{\code{temporal_grp_periodicity > frequency(in_ts)}} start on a
year that is a multiple of \ifelse{latex}{\code{ceiling(temporal_grp _periodicity / frequency(in_ts))}}{\code{
ceiling(temporal_grp_periodicity / frequency(in_ts))}}. E.g., biennial groups corresponding to \ifelse{latex}{
\code{temporal_grp_per iodicity = 2 * frequency(in_ts)}}{\code{temporal_grp_periodicity = 2 * frequency(in_ts)}}
start on an \emph{even year} by default. This behaviour can be changed with argument \code{temporal_grp_start}. E.g., the
preservation of biennial totals starting on an \emph{odd year} instead of an \emph{even year} (default) corresponds to
\code{temporal_grp_start = frequency(in_ts) + 1} (along with \code{temporal_grp_periodicity = 2 * frequency(in_ts)}).

See the \code{\link[=gs.build_proc_grps]{gs.build_proc_grps()}} \strong{Examples} for common processing group scenarios.
}

\examples{
#######
# Preliminary setup

# Dummy monthly and quarterly time series (2.5 years long)
mth_ts <- ts(rep(NA, 30), start = c(2019, 1), frequency = 12)
mth_ts
qtr_ts <- ts(rep(NA, 10), start = c(2019, 1), frequency = 4)
qtr_ts

# Summarized time series info
ts_info <- function(ts, sep = "-") {
  list(y = gs.time2year(ts),      # years
       p = gs.time2per(ts),       # periods
       n = length(ts),            # length
       f = frequency(ts),         # frequency
       l = gs.time2str(ts, sep))  # labels
}
mth_info <- ts_info(mth_ts)
qtr_info <- ts_info(qtr_ts, sep = "q")

# Function to add a description label for the processing group
add_desc <- function(grp_df, lab_vec, word) {
  grp_df$description <- ifelse(grp_df$complete_grp,
                               paste0("--- ", grp_df$end_per - grp_df$beg_per + 1, " ", word, "s: ",
                                      lab_vec[grp_df$beg_per], " to ",
                                      lab_vec[grp_df$end_per], " --- "),
                               paste0("--- 1 ", word, ": ", lab_vec[grp_df$beg_per], " ---"))
  grp_df
}




#######
# Common processing group scenarios for monthly data


# 0- Month-by-month processing (every single month is a processing group)
mth_grps0 <- gs.build_proc_grps(mth_info$y, mth_info$p, mth_info$n, mth_info$f,
                                temporal_grp_periodicity = 1,
                                temporal_grp_start = 1)
tmp <- add_desc(mth_grps0, mth_info$l, "month")
head(tmp)
tail(tmp)


# Temporal groups corresponding to ...

# 1- calendar years
mth_grps1 <- gs.build_proc_grps(mth_info$y, mth_info$p, mth_info$n, mth_info$f,
                                temporal_grp_periodicity = 12,
                                temporal_grp_start = 1)
add_desc(mth_grps1, mth_info$l, "month")

# 2- fiscal years starting on April
mth_grps2 <- gs.build_proc_grps(mth_info$y, mth_info$p, mth_info$n, mth_info$f,
                                temporal_grp_periodicity = 12,
                                temporal_grp_start = 4)
add_desc(mth_grps2, mth_info$l, "month")

# 3- regular quarters (starting on Jan, Apr, Jul and Oct)
mth_grps3 <- gs.build_proc_grps(mth_info$y, mth_info$p, mth_info$n, mth_info$f,
                                temporal_grp_periodicity = 3,
                                temporal_grp_start = 1)
add_desc(mth_grps3, mth_info$l, "month")

# 4- quarters shifted by one month (starting on Feb, May, Aug and Nov)
mth_grps4 <- gs.build_proc_grps(mth_info$y, mth_info$p, mth_info$n, mth_info$f,
                                temporal_grp_periodicity = 3,
                                temporal_grp_start = 2)
add_desc(mth_grps4, mth_info$l, "month")




#######
# Common processing group scenarios for quarterly data


# 0- Quarter-by-quarter processing (every single quarter is a processing group)
qtr_grps0 <- gs.build_proc_grps(qtr_info$y, qtr_info$p, qtr_info$n, qtr_info$f,
                                temporal_grp_periodicity = 1,
                                temporal_grp_start = 1)
add_desc(qtr_grps0, qtr_info$l, "quarter")


# Temporal groups corresponding to ...

# 1- calendar years
qtr_grps1 <- gs.build_proc_grps(qtr_info$y, qtr_info$p, qtr_info$n, qtr_info$f,
                                temporal_grp_periodicity = 4,
                                temporal_grp_start = 1)
add_desc(qtr_grps1, qtr_info$l, "quarter")

# 2- fiscal years starting on April (2nd quarter)
qtr_grps2 <- gs.build_proc_grps(qtr_info$y, qtr_info$p, qtr_info$n, qtr_info$f,
                                temporal_grp_periodicity = 4,
                                temporal_grp_start = 2)
add_desc(qtr_grps2, qtr_info$l, "quarter")
}
\seealso{
\code{\link[=tsraking_driver]{tsraking_driver()}} \code{\link[=tsbalancing]{tsbalancing()}} \link{time_values_conv}
}
