#' Util fuunction to generate haplotype dataframe from raw data.
#'
#' @param in_df Data frame of raw data.
#' @param target Character of genotype which is the target of gene pyramiding.
#' @param non_target Character of genotype which is not the target of gene pyramiding.
#' @param hetero Character of genotype of heterozygote.
#' @param line_cul Column name containing line identifiers.
#'
#' @returns A list of genotype and line names for downstream analysis. This is a list with the following components:
#' * `gene_df1` (`data frame`): One set of haplotype. Values take 1 (target allele) or 0 (non-target).
#' * `gene_df2` (`data frame`): The other set of haplotype. Values take 1 or 0.
#' * `line_id` (`vector`): Line names.
#' @examples
#' in_df <- data.frame(line = c("CV1", "CV2", "CV3", "CV4", "CV5"),
#'                     gene1 = c("A", "A", "B", "B", "A"),
#'                     gene2 = c("B", "A", "A", "B", "H"),
#'                     gene3 = c("A", "A", "H", "H", "A"),
#'                     gene4 = c("A", "B", "H", "A", "B"))
#'
#' util_haplo(in_df, target = "A", non_target = "B", hetero = "H", line_cul = "line")
#'
#' @export

util_haplo <- function(in_df, target, non_target, hetero, line_cul){
  line_id <- in_df[, line_cul]
  in_df <- in_df[, !colnames(in_df) == line_cul]
  gene_df1 <- matrix(0, nrow = nrow(in_df), ncol = ncol(in_df))
  gene_df2 <- matrix(0, nrow = nrow(in_df), ncol = ncol(in_df))

  for(i in 1:nrow(in_df)){
    for(j in 1:ncol(in_df)){
      if(in_df[i, j] == target){
        gene_df1[i, j] <- 1
        gene_df2[i, j] <- 1
      }else if(in_df[i, j] == hetero){
        gene_df1[i, j] <- 1
      }
    }
  }

  gene_df1 <- t(gene_df1)
  gene_df2 <- t(gene_df2)

  return(list(gene_df1, gene_df2, line_id))
}

#' Util function to generate recom_mat from raw data.
#' It returens the recombination probability based on Halden's map function
#'
#' @param in_df Data frame of raw data. The column names should be "Gene", "Chr" and "cM".
#' @param unit Unit of the gene positions. In the current version, it should be "cM".
#'
#' @returns util_recom_mat function returns a matrix of recombination probability between each combination of genes.
#'
#' @references Haldane (1919) The combination of linkage values, and the calculation of distances between the loci of linked factors.
#' Journal of Genetics 8: 299-309.
#' @examples
#'
#' in_df <- data.frame(Gene = c("g1", "g2", "g3", "g4"),
#'                     Chr = c("1A", "1B", "2A", "2A"),
#'                     cM = c(30, 50, 35, 50))
#'
#' util_recom_mat(in_df)
#'
#' @export


util_recom_mat <- function(in_df, unit = "cM"){
  n_gene <- nrow(in_df)
  recom_mat <- matrix(0, nrow = n_gene, ncol = n_gene)

  for(i in 1:n_gene){
    for(j in 1:n_gene){
      if(i != j){
        if(in_df$Chr[i] != in_df$Chr[j]){
            recom_mat[i, j] <- 0.5
        }else{
          cM_diff <- abs(in_df$cM[i] - in_df$cM[j])
          recom_mat[i, j] <- 1/2 * (1 - exp(-2 * cM_diff/100))
        }
      }
    }
  }
  return(recom_mat)
}


