% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/facet_geo.R
\name{facet_geo}
\alias{facet_geo}
\title{Arrange a sequence of geographical panels into a grid that preserves some geographical orientation}
\usage{
facet_geo(facets, ..., grid = "us_state_grid1", label = NULL, move_axes = TRUE)
}
\arguments{
\item{facets}{passed to \code{\link[ggplot2]{facet_wrap}}}

\item{grid}{either a character vector of the grid layout to use (see `?grids` for the list and use `get_grid()` to inspect or `grid_preview()` to plot a specific grid),
or a data.frame object containing a grid (e.g. an output from `grid_design()` or `grid_auto()`)}

\item{label}{an optional string denoting the name of a column in \code{grid} to use for facet labels. If NULL, the variable that best matches that in the data specified with \code{facets} will be used for the facet labels.}

\item{move_axes}{should axis labels and ticks be moved to the closest panel along the margins?}

\item{\ldots}{additional parameters passed to \code{\link[ggplot2]{facet_wrap}}}
}
\description{
Arrange a sequence of geographical panels into a grid that preserves some geographical orientation
}
\examples{
\dontrun{
library(ggplot2)

# barchart of state rankings in various categories
ggplot(state_ranks, aes(variable, rank, fill = variable)) +
  geom_col() +
  coord_flip() +
  facet_geo(~ state) +
  theme_bw()

# use an alternative US state grid and place
ggplot(state_ranks, aes(variable, rank, fill = variable)) +
  geom_col() +
  coord_flip() +
  facet_geo(~ state, grid = "us_state_grid2") +
  theme(panel.spacing = unit(0.1, "lines"))

# custom grid (move Wisconsin above Michigan)
my_grid <- us_state_grid1
my_grid$col[my_grid$code == "WI"] <- 7

ggplot(state_ranks, aes(variable, rank, fill = variable)) +
  geom_col() +
  coord_flip() +
  facet_geo(~ state, grid = my_grid)

# plot unemployment rate time series for each state
ggplot(state_unemp, aes(year, rate)) +
  geom_line() +
  facet_geo(~ state) +
  scale_x_continuous(labels = function(x) paste0("'", substr(x, 3, 4))) +
  ylab("Unemployment Rate (\%)") +
  theme_bw()

# plot the 2016 unemployment rate
ggplot(subset(state_unemp, year == 2016), aes(factor(year), rate)) +
  geom_col(fill = "steelblue") +
  facet_geo(~ state) +
  theme(
    axis.title.x = element_blank(),
    axis.text.x = element_blank(),
    axis.ticks.x = element_blank()) +
  ylab("Unemployment Rate (\%)") +
  xlab("Year")

# plot European Union GDP
ggplot(eu_gdp, aes(year, gdp_pc)) +
  geom_line(color = "steelblue") +
  geom_hline(yintercept = 100, linetype = 2) +
  facet_geo(~ name, grid = "eu_grid1") +
  scale_x_continuous(labels = function(x) paste0("'", substr(x, 3, 4))) +
  ylab("GDP Per Capita") +
  theme_bw()

# use a free x-axis to look at just change
ggplot(eu_gdp, aes(year, gdp_pc)) +
  geom_line(color = "steelblue") +
  facet_geo(~ name, grid = "eu_grid1", scales = "free_y") +
  scale_x_continuous(labels = function(x) paste0("'", substr(x, 3, 4))) +
  ylab("GDP Per Capita in Relation to EU Index (100)") +
  theme_bw()
# would be nice if ggplot2 had a "sliced" option...
# (for example, there's not much going on with Denmark but it looks like there is)

# plot European Union annual # of resettled persons
ggplot(eu_imm, aes(year, persons)) +
  geom_line() +
  facet_geo(~ name, grid = "eu_grid1") +
  scale_x_continuous(labels = function(x) paste0("'", substr(x, 3, 4))) +
  scale_y_sqrt(minor_breaks = NULL) +
  ylab("# Resettled Persons") +
  theme_bw()

# plot just for 2016
ggplot(subset(eu_imm, year == 2016), aes(factor(year), persons)) +
  geom_col(fill = "steelblue") +
  geom_text(aes(factor(year), 3000, label = persons), color = "gray") +
  facet_geo(~ name, grid = "eu_grid1") +
  theme(
    axis.title.x = element_blank(),
    axis.text.x = element_blank(),
    axis.ticks.x = element_blank()) +
  ylab("# Resettled Persons in 2016") +
  xlab("Year") +
  theme_bw()

# plot Australian population
ggplot(aus_pop, aes(age_group, pop / 1e6, fill = age_group)) +
  geom_col() +
  facet_geo(~ code, grid = "aus_grid1") +
  coord_flip() +
  labs(
    title = "Australian Population Breakdown",
    caption = "Data Source: ABS Labour Force Survey, 12 month average",
    y = "Population [Millions]") +
  theme_bw()

# South Africa population density by province
ggplot(sa_pop_dens, aes(factor(year), density, fill = factor(year))) +
  geom_col() +
  facet_geo(~ province, grid = "sa_prov_grid1") +
  labs(title = "South Africa population density by province",
    caption = "Data Source: Statistics SA Census",
    y = "Population density per square km") +
  theme_bw()

# use the Afrikaans name stored in the grid, "name_af", as facet labels
ggplot(sa_pop_dens, aes(factor(year), density, fill = factor(year))) +
  geom_col() +
  facet_geo(~ code, grid = "sa_prov_grid1", label = "name_af") +
  labs(title = "South Africa population density by province",
    caption = "Data Source: Statistics SA Census",
    y = "Population density per square km") +
  theme_bw()

# affordable housing starts by year for boroughs in London
ggplot(london_afford, aes(x = year, y = starts, fill = year)) +
  geom_col(position = position_dodge()) +
  facet_geo(~ code, grid = "london_boroughs_grid", label = "name") +
  labs(title = "Affordable Housing Starts in London",
    subtitle = "Each Borough, 2015-16 to 2016-17",
    caption = "Source: London Datastore", x = "", y = "")

# dental health in Scotland
ggplot(nhs_scot_dental, aes(x = year, y = percent)) +
  geom_line() +
  facet_geo(~ name, grid = "nhs_scot_grid") +
  scale_x_continuous(breaks = c(2004, 2007, 2010, 2013)) +
  scale_y_continuous(breaks = c(40, 60, 80)) +
  labs(title = "Child Dental Health in Scotland",
    subtitle = "Percentage of P1 children in Scotland with no obvious decay experience.",
    caption = "Source: statistics.gov.scot", x = "", y = "")

# India population breakdown
ggplot(subset(india_pop, type == "state"),
  aes(pop_type, value / 1e6, fill = pop_type)) +
  geom_col() +
  facet_geo(~ name, grid = "india_grid1", label = "code") +
  labs(title = "Indian Population Breakdown",
       caption = "Data Source: Wikipedia",
       x = "",
       y = "Population [Millions]") +
  theme_bw() +
  theme(axis.text.x = element_text(angle = 40, hjust = 1))

ggplot(subset(india_pop, type == "state"),
  aes(pop_type, value / 1e6, fill = pop_type)) +
  geom_col() +
  facet_geo(~ name, grid = "india_grid2", label = "name") +
  labs(title = "Indian Population Breakdown",
       caption = "Data Source: Wikipedia",
       x = "",
       y = "Population [Millions]") +
  theme_bw() +
  theme(axis.text.x = element_text(angle = 40, hjust = 1),
    strip.text.x = element_text(size = 6))

# A few ways to look at the 2016 election results
ggplot(election, aes("", pct, fill = candidate)) +
  geom_col(alpha = 0.8, width = 1) +
  scale_fill_manual(values = c("#4e79a7", "#e15759", "#59a14f")) +
  facet_geo(~ state, grid = "us_state_grid2") +
  scale_y_continuous(expand = c(0, 0)) +
  labs(title = "2016 Election Results",
    caption = "Data Source: 2016 National Popular Vote Tracker",
    x = NULL,
    y = "Percentage of Voters") +
  theme(axis.title.x = element_blank(),
    axis.text.x = element_blank(),
    axis.ticks.x = element_blank(),
    strip.text.x = element_text(size = 6))

ggplot(election, aes(candidate, pct, fill = candidate)) +
  geom_col() +
  scale_fill_manual(values = c("#4e79a7", "#e15759", "#59a14f")) +
  facet_geo(~ state, grid = "us_state_grid2") +
  theme_bw() +
  coord_flip() +
  labs(title = "2016 Election Results",
    caption = "Data Source: 2016 National Popular Vote Tracker",
    x = NULL,
    y = "Percentage of Voters") +
  theme(strip.text.x = element_text(size = 6))

ggplot(election, aes(candidate, votes / 1000000, fill = candidate)) +
  geom_col() +
  scale_fill_manual(values = c("#4e79a7", "#e15759", "#59a14f")) +
  facet_geo(~ state, grid = "us_state_grid2") +
  coord_flip() +
  labs(title = "2016 Election Results",
    caption = "Data Source: 2016 National Popular Vote Tracker",
    x = NULL,
    y = "Votes (millions)") +
  theme(strip.text.x = element_text(size = 6))
}
}
