% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/time_lock_stats.R
\name{time_lock_stats}
\alias{time_lock_stats}
\title{Calculate Statistics for Time Locked Bins}
\usage{
time_lock_stats(
  fp.dt,
  vars,
  time.lock.trigger,
  bins,
  bin.width = NULL,
  n.bins = NULL,
  FUN = list(mean = mean, sd = sd, range = function(x) diff(range(x)))
)
}
\arguments{
\item{fp.dt}{A \code{data.table} of the class \code{"fp.segm"} produced by \code{segment_fp_data()}.}

\item{vars}{A (vector of) character(s) giving the variable names in \code{fp.dt$forceplate}, for
which the statistics (see \code{FUN} below) should be calculated for each bin (see arguments below). For example Fx, Fy, Mx, My etc.}

\item{time.lock.trigger}{A (vector of) number(s) containing the trigger(s) marking the onset of the
time locking (the event of interest like stimulus onset or response onset). The onset of this trigger will be treated as reference (point zero) for the bins to
be defined in the next argument(s).}

\item{bins}{Either a vector of length 2 or a list of vectors of length 2 providing the lower and
upper boundaries of the bins (in milliseconds). If only one vector is used either one of the next
two arguments can be used to make (equaly sized) bins. If a list is used the next two arguments
are ignored.}

\item{bin.width}{If \code{bins} is a vector of 2 then this argument can be used to divide the bin
into smaller bins of the size \code{bin.width} in milliseconds.}

\item{n.bins}{If \code{bins} is a vector of 2 then this argument can be used to divide the bin into
\code{n.bins} number of bins of equal size. If \code{bin.width} is provided as well, \code{n.bins}
will be ignored.}

\item{FUN}{A list of functions. These functions should be statistics that take as input a vector and
return a scalar. See usage for an example (mean, standard deviation, range).}
}
\value{
A \code{data.table} of the class \code{fp.tl}.
The following variables are included in the \code{data.table}:
\itemize{
\item \code{subj}: subject number,
\item \code{block}: block number,
\item \code{trial}: trial number,
\item \code{forceplate}: force-plate data of each trial as \code{data.table}. Use, for example,
\code{fp.dt$forceplate[[1]]} to open the force-plate data of the first trial, first block, and first subject
(if \code{sort} in the \code{\link{segment_fp_data}} was set to \code{TRUE}.
\item For each combination of variable \code{vars} and \code{bin} a new variable is created by the function(s) provided
by \code{FUN}.
}
}
\description{
Processing segmented force-plate data by calculating descriptive statistics like mean, standard
deviation, and range for defined time bins around time-locked events, such as stimulus onset
or response onset etc.
}
\examples{
# Using example data from github which requires internet
\donttest{ # takes longer than 5 seconds
if (curl::has_internet()) {
  url <- paste0("https://raw.githubusercontent.com/RaphaelHartmann/forceplate/",
                "main/data/subj013_block001.txt")
  
  # Safe download, handling potential errors
  tryCatch({
    filenames <- tempfile(pattern = c("subj013_block001_"), 
                          tmpdir = tempdir(), fileext = ".txt")
    download.file(url, filenames)
    fp.dt <- segment_fp_data(filenames = filenames, n.trials = 80, baseline.trigger = 128,
                             baseline.intv = c(0, 215), start.trigger = 128, start.prepend = 0,
                             stimulus.trigger.list = c(1, 2, 4, 8),
                             response.trigger.list = c(32, 64),
                             cond.trigger.list = list(stimulus = c(1, 2, 4, 8), 
                                                      correctness = c(32, 64)))
    
    # Response-locking with 2 bins before and 2 bins after response onset. Each bin is 100 ms.
    tl.dt <- time_lock_stats(fp.dt = fp.dt, vars = c("Mx", "My"), 
                             time.lock.trigger = c(1,2,4,8), bins = c(-150, 150), n.bins = 2, 
                             FUN = list(mean = mean, sd = sd, range = function(x) diff(range(x))))
    
    # Clean up
    unlink(filenames)
  }, error = function(e) {
    message("Failed to download data: ", e$message)
  })
}
}

}
\references{
Johannsen, L., Stephan, D. N., Straub, E., Döhring, F., Kiesel, A., Koch, I., & Müller, H. (2023). Assessing the influence of cognitive response conflict on balance control: An
event-related approach using response-aligned force-plate time series data.
\emph{Psychological Research, 87}, 2297–2315.
}
\author{
Raphael Hartmann & Anton Koger
}
