% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/simulations.R
\name{SimulateAdjacency}
\alias{SimulateAdjacency}
\title{Simulation of undirected graph with block structure}
\usage{
SimulateAdjacency(
  pk = 10,
  implementation = HugeAdjacency,
  topology = "random",
  nu_within = 0.1,
  nu_between = 0,
  nu_mat = NULL,
  ...
)
}
\arguments{
\item{pk}{vector of the number of variables per group in the simulated
dataset. The number of nodes in the simulated graph is \code{sum(pk)}. With
multiple groups, the simulated (partial) correlation matrix has a block
structure, where blocks arise from the integration of the \code{length(pk)}
groups. This argument is only used if \code{theta} is not provided.}

\item{implementation}{function for simulation of the graph. By default,
algorithms implemented in \code{\link[igraph]{igraph}}
are used. Alternatively, a user-defined function can be used. It must take
\code{pk}, \code{topology} and \code{nu} as arguments and return a
\code{(sum(pk)*(sum(pk)))} binary and symmetric matrix for which diagonal
entries are all equal to zero. This function is only applied if
\code{theta} is not provided.}

\item{topology}{topology of the simulated graph. Possible arguments are:
"random", or "scale-free".}

\item{nu_within}{probability of having an edge between two nodes belonging to
the same group, as defined in \code{pk}. If \code{length(pk)=1}, this is
the expected density of the graph. If \code{implementation=HugeAdjacency},
this argument is only used for \code{topology="random"}. Only used if
\code{nu_mat} is not provided.}

\item{nu_between}{probability of having an edge between two nodes belonging
to different groups, as defined in \code{pk}. By default, the same density
is used for within and between blocks (\code{nu_within}=\code{nu_between}).
Only used if \code{length(pk)>1}. Only used if \code{nu_mat} is not
provided.}

\item{nu_mat}{matrix of probabilities of having an edge between nodes
belonging to a given pair of node groups defined in \code{pk}.}

\item{...}{additional arguments passed to the graph simulation function
provided in \code{implementation}.}
}
\value{
A symmetric adjacency matrix encoding an unweighted, undirected graph
  with no self-loops, and with different densities in diagonal compared to
  off-diagonal blocks.
}
\description{
Simulates the adjacency matrix of an unweighted, undirected graph with no
self-loops. If \code{topology="random"}, different densities in diagonal
(\code{nu_within}) compared to off-diagonal (\code{nu_between}) blocks can be
used.
}
\details{
Random graphs are simulated using the Erdos-Renyi algorithm.
  Scale-free graphs are simulated using a preferential attachment algorithm.
}
\examples{
# Simulation of a scale-free graph with 20 nodes
adjacency <- SimulateAdjacency(pk = 20, topology = "scale-free")
plot(adjacency)

# Simulation of a random graph with three connected components
adjacency <- SimulateAdjacency(
  pk = rep(10, 3),
  nu_within = 0.7, nu_between = 0
)
plot(adjacency)

# Simulation of a random graph with block structure
adjacency <- SimulateAdjacency(
  pk = rep(10, 3),
  nu_within = 0.7, nu_between = 0.03
)
plot(adjacency)

# User-defined function for graph simulation
CentralNode <- function(pk, hub = 1) {
  theta <- matrix(0, nrow = sum(pk), ncol = sum(pk))
  theta[hub, ] <- 1
  theta[, hub] <- 1
  diag(theta) <- 0
  return(theta)
}
simul <- SimulateAdjacency(pk = 10, implementation = CentralNode)
plot(simul) # star
simul <- SimulateAdjacency(pk = 10, implementation = CentralNode, hub = 2)
plot(simul) # variable 2 is the central node
}
\references{
\insertRef{ourstabilityselection}{fake}

\insertRef{huge}{fake}
}
\seealso{
Other simulation functions: 
\code{\link{SimulateClustering}()},
\code{\link{SimulateComponents}()},
\code{\link{SimulateCorrelation}()},
\code{\link{SimulateGraphical}()},
\code{\link{SimulateRegression}()},
\code{\link{SimulateStructural}()}
}
\concept{simulation functions}
