% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/eratosthenes.R
\name{sq_disp}
\alias{sq_disp}
\alias{sq_disp.marginals}
\title{Squared Displacement for a Target Event}
\usage{
sq_disp(
  marginalized,
  target,
  sequences,
  finds = NULL,
  max_samples = 10^5,
  size = 10^3,
  mcse_crit = 0.5,
  tpq = NULL,
  taq = NULL,
  alpha_ = -5000,
  omega_ = 1950,
  rule = "naive"
)

\method{sq_disp}{marginals}(
  marginalized,
  target,
  sequences,
  finds = NULL,
  max_samples = 10^5,
  size = 10^3,
  mcse_crit = 0.5,
  tpq = NULL,
  taq = NULL,
  alpha_ = -5000,
  omega_ = 1950,
  rule = "naive"
)
}
\arguments{
\item{marginalized}{The results of \code{\link[eratosthenes]{gibbs_ad}}.\code{\link[eratosthenes]{gibbs_ad}}.}

\item{target}{The target event (any event in \code{marginalized}) for which to estimate squared displacement.}

\item{sequences}{A \code{list} of relative sequences of elements (e.g., contexts) used to compute \code{marginalized}.}

\item{finds}{Optional. A \code{list} of finds related to (contained in) the elements of \code{sequences}.}

\item{max_samples}{Maximum number of samples to run. Default is \code{10^5}.}

\item{size}{The number of samples to take on each iteration of the main Gibbs sampler. Default is \code{10^3}.}

\item{mcse_crit}{Criterion for the Monte Carlo standard error to stop the Gibbs sampler. A higher MCSE is recommended for situations with a higher number of events in order to reduce computational time.}

\item{tpq}{A \code{list} containing \emph{termini post quos} used to compute \code{marginalized}. See \code{\link[eratosthenes]{gibbs_ad}} for details.}

\item{taq}{A \code{list} containing \emph{termini ante quos} used to compute \code{marginalized}. See \code{\link[eratosthenes]{gibbs_ad}} for details.}

\item{alpha_}{An initial \emph{t.p.q.} to limit any elements which may occur before the first provided \emph{t.p.q.} Default is \code{-5000}.}

\item{omega_}{A final \emph{t.a.q.} to limit any elements which may occur after the after the last provided \emph{t.a.q.} Default is \code{1950}.}

\item{rule}{The rule for computing an estimated date of production. See \code{\link[eratosthenes]{gibbs_ad}} for details.}
}
\value{
Output is a list containing a data frame \code{sq_disp} giving the diplacement with respect to all other events and a vector \code{bounds} of the values of \code{alpha_} and \code{omega_}.
}
\description{
Computes the squared displacement for a target event within the joint conditional density, estimating how much the omission of another event will change the date of that event. See also \code{\link[eratosthenes]{msd}}.
}
\details{
Displacement is computed via the following jackknife/leave-one-out-style routine:
\itemize{
\item Each event, excluding the target event itself, is omitted from all relative and absolute sequences, and the function \code{\link[eratosthenes]{gibbs_ad}} is re-run to compute a "jackknifed" Monte Carlo mean for the target event.
\itemize{
\item The squared difference of this jackknifed Monte Carlo mean and the original is then computed as its squared "displacement" in time.
}
}

If an event has a low squared displacement, it has a low impact on the dating of the target event. If it is has a high squared displacement, the target event's date depends heavily upon its inclusion in the full joint density.

Trimming is not implemented in the estimation of squared displacement, and so attention should be paid to the selection of \code{alpha_} and \code{omega_}, and reported. This is owing to the way in which, if an absolute constraint (\code{tpq} or \code{taq}) is omitted that happens to be an earliest or latest bounding event, there still needs to be earliest and latest thresholds in place.

This function is fairly computationally intensive, and so a lower value of \code{max_samples} or higher value of \code{mcse_crit} may be warranted.
}
\examples{
x <- c("A", "B", "C", "D", "E", "F", "G", "H", "I", "J")
y <- c("B", "D", "G", "H", "K")
z <- c("F", "K", "L", "M")
contexts <- list(x, y, z)

f1 <- list(id = "find01", assoc = "D", type = c("type1", "form1"))
f2 <- list(id = "find02", assoc = "E", type = c("type1", "form2"))
f3 <- list(id = "find03", assoc = "G", type = c("type1", "form1"))
f4 <- list(id = "find04", assoc = "H", type = c("type2", "form1"))
f5 <- list(id = "find05", assoc = "I", type = "type2")
f6 <- list(id = "find06", assoc = "H", type = NULL)

artifacts <- list(f1, f2, f3, f4, f5, f6)
 
# external constraints
coin1 <- list(id = "coin1", assoc = "B", type = NULL, samples = runif(100,-320,-300))
coin2 <- list(id = "coin2", assoc = "G", type = NULL, samples = seq(37, 41, length = 100))
destr <- list(id = "destr", assoc = "J", type = NULL, samples = 79)

tpq_info <- list(coin1, coin2)
taq_info <- list(destr)

result <- gibbs_ad(contexts, finds = artifacts, tpq = tpq_info, taq = taq_info)

# max_samples lowered and msce_crit raised for examples

# squared displacement for depositional context "E"
E_sqdisp <- sq_disp(result, target = "E", sequences = contexts, 
                    max_samples = 3000, mcse_crit = 2, tpq = tpq_info, taq = taq_info)

# squared displacement for production of artifact type "type1"
type1_sqdisp <- sq_disp(result, target = "type1", sequences = contexts, finds = artifacts,
                        max_samples = 3000, mcse_crit = 2, tpq = tpq_info, taq = taq_info)

}
