set.seed(730)

##### Top Tests

test_that("formals same as switch", {
  b <- names(as.list(body(char_to_dist))[[2]][-c(1, 2)])
  b <- b[b != ""] # drop the ""
  f <- unlist(as.list(formals(energyGOF.test.character)$y)[-1])
  expect_setequal(b, f)
})


#### Idea; tests that ensure the p-value distributions are uniform under Null.

##### test that distributions are correctly formed

test_that("Dists", {
  ## F
  expect_s3_class(f_dist(4, 4), "GOFDist")
  expect_error(f_dist(4, NULL))
  expect_error(f_dist(NULL, 4))
  expect_error(f_dist(NULL, NULL))
  expect_error(f_dist(2, 2))
  ## Uniform
  expect_s3_class(uniform_dist(0, 1), "GOFDist")
  expect_s3_class(uniform_dist(-100, 100), "GOFDist")
  expect_error(uniform_dist(100, -100))
  ## Normal
  expect_s3_class(normal_dist(0, 1), "GOFDist")
  expect_s3_class(normal_dist(), "GOFDist")
  expect_error(normal_dist(0, -1))
  expect_error(normal_dist(0, NULL))
  expect_error(normal_dist(NULL, 0))
  ## Exp
  expect_s3_class(exponential_dist(1), "GOFDist")
  expect_s3_class(exponential_dist(), "GOFDist")
  expect_error(exponential_dist(-1))
  ## Poisson
  expect_s3_class(poisson_dist(1), "GOFDist")
  expect_s3_class(poisson_dist(), "GOFDist")
  expect_error(poisson_dist(-1))
  ## Bernoulli
  expect_s3_class(bernoulli_dist(.5), "GOFDist")
  expect_s3_class(bernoulli_dist(), "GOFDist")
  expect_error(bernoulli_dist(-1))
  ## Binomial
  expect_s3_class(binomial_dist(5, .5), "GOFDist")
  expect_s3_class(binomial_dist(5), "GOFDist")
  expect_error(binomial_dist(NULL, 5))
  expect_error(binomial_dist(5, NULL))
  expect_error(binomial_dist(-1))
  ## Beta
  expect_s3_class(beta_dist(1, 1), "GOFDist")
  expect_s3_class(beta_dist(1e6, 1 / 1e5), "GOFDist")
  expect_error(beta_dist(1, NULL))
  expect_error(beta_dist(NULL, 1))
  expect_s3_class(beta_dist(), "GOFDist")
  ## Geometric
  expect_s3_class(geometric_dist(.5), "GOFDist")
  expect_s3_class(geometric_dist(), "GOFDist")
  expect_error(geometric_dist(-1))
  ## Half normal
  expect_s3_class(halfnormal_dist(.5), "GOFDist")
  expect_s3_class(halfnormal_dist(), "GOFDist")
  expect_error(halfnormal_dist(-1))
  ## Laplace
  expect_s3_class(laplace_dist(0, 1), "GOFDist")
  expect_s3_class(laplace_dist(), "GOFDist")
  expect_error(laplace_dist(0, -1))
  expect_error(laplace_dist(0, NULL))
  expect_error(laplace_dist(NULL, 0))
  ## Lognormal
  expect_s3_class(lognormal_dist(0, 1), "GOFDist")
  expect_s3_class(lognormal_dist(), "GOFDist")
  expect_error(lognormal_dist(0, -1))
  expect_error(lognormal_dist(0, NULL))
  expect_error(lognormal_dist(NULL, 0))
  ## Asym Laplace
  expect_s3_class(asymmetric_laplace_dist(0, 1, 1), "GOFDist")
  expect_s3_class(asymmetric_laplace_dist(), "GOFDist")
  expect_error(asymmetric_laplace_dist(1, 1, -1))
  expect_error(asymmetric_laplace_dist(1, -1, 1))
  expect_error(asymmetric_laplace_dist(NULL, -1, 1))
  expect_error(asymmetric_laplace_dist(1, NULL, 1))
  expect_error(asymmetric_laplace_dist(1, 1, NULL))
  ## Weibull
  expect_s3_class(weibull_dist(1, 1), "GOFDist")
  expect_s3_class(weibull_dist(), "GOFDist")
  expect_error(weibull_dist(1, NULL))
  expect_error(weibull_dist(NULL, 1))
  expect_error(weibull_dist(-1, 1))
  expect_error(weibull_dist(1, -1))
  ## Gamma
  expect_s3_class(gamma_dist(1, 1), "GOFDist")
  expect_s3_class(gamma_dist(), "GOFDist")
  expect_error(gamma_dist(1, NULL))
  expect_error(gamma_dist(NULL, 1))
  expect_error(gamma_dist(-1, 1))
  expect_error(gamma_dist(1, -1))
  ## Chi-sq
  expect_s3_class(chisq_dist(1), "GOFDist")
  expect_s3_class(chisq_dist(), "GOFDist")
  expect_error(chisq_dist(NULL))
  ## Inv Gaussian
  expect_s3_class(inverse_gaussian_dist(1, 1), "GOFDist")
  expect_s3_class(inverse_gaussian_dist(), "GOFDist")
  expect_error(inverse_gaussian_dist(1, NULL))
  expect_error(inverse_gaussian_dist(NULL, 1))
  expect_error(inverse_gaussian_dist(-1, 1))
  expect_error(inverse_gaussian_dist(1, -1))
  ## Pareto
  expect_s3_class(pareto_dist(1, 1), "GOFDist")
  expect_s3_class(pareto_dist(1e5, 1 / 1e5), "GOFDist")
  expect_s3_class(pareto_dist(1 / 1e5, 1e5), "GOFDist")
  expect_s3_class(pareto_dist(1e5, 1e5), "GOFDist") # Rounds to zero!
  expect_s3_class(pareto_dist(5, 5, 1), "GOFDist")
  expect_error(pareto_dist(5, 5, 2))
  expect_error(pareto_dist(5, 5, 3))
  expect_error(pareto_dist(5, -5, 2))
  expect_error(pareto_dist(-5, 5, 2))
  expect_error(pareto_dist(NULL, 4, 2))
  expect_s3_class(pareto_dist(NULL, NULL, 2), "GOFDist")
  expect_s3_class(pareto_dist(NULL, NULL, NULL), "GOFDist")
  expect_error(pareto_dist(4, NULL, 2))
  expect_s3_class(pareto_dist(4, 4, NULL), "GOFDist")
  ## Cauchy
  expect_s3_class(cauchy_dist(0, 1), "GOFDist")
  expect_s3_class(cauchy_dist(), "GOFDist")
  expect_error(cauchy_dist(0, -1))
  expect_error(cauchy_dist(0, NULL))
  expect_error(cauchy_dist(NULL, 0))
  ## Stable
  ## expect_s3_class(stable_dist(), "GOFDist")
  ## expect_error(stable_dist(NULL, 1, 1, 1))
  ## expect_error(stable_dist(0, NULL, 1, 1))
  ## expect_error(stable_dist(0, 1, NULL, 1))
  ## expect_error(stable_dist(0, 0, 1, NULL))
  ## expect_error(stable_dist(0, 0, 0, 0))
  ## expect_error(stable_dist(1, 1, 1, 1, 1))
})

##### egof.test tests

test_that("egof.test", {
  x <- rnorm(10)
  expect_s3_class(egof.test(x, "normal", 0), "htest")
  expect_s3_class(egof.test(x, "laplace", 0), "htest")
  expect_s3_class(egof.test(x, "cauchy", 0), "htest")
  expect_s3_class(egof.test(x, x, 0), "htest")
  expect_s3_class(egof.test(x, pnorm, 0), "htest")
})

##### egof.test.function tests

test_that("function method tests", {
  x <- rnorm(10)
  y <- rexp(10)
  expect_s3_class(egof.test(x, pnorm, 0), "htest")
  expect_s3_class(egof.test(y, pexp, 0), "htest")
  expect_s3_class(egof.test(x, pt, 0, df = 5), "htest")
  expect_s3_class(egof.test(x, pcauchy, 0, location = 5, scale = 1), "htest")
  expect_warning(egof.test(x, pexp, 0, rate = 1))
  expect_warning(
    ## Two warnings, so two calls
    expect_warning(egof.test(x, pbinom, 0, size = 3, prob = 0.4))
  )
})

##### egof.test.numeric tests

test_that("numeric tests", {
  x <- rnorm(10)
  y <- rt(10, 5)
  expect_s3_class(egof.test(x, y, 0), "htest")
})

##### Tests for Validate_CDF

test_that("validate_cdf works", {
  expect_no_error(validate_cdf(pexp, rexp(10)))
  expect_no_error(validate_cdf(pt, rt(10, 5), df = 5))
  expect_no_error(validate_cdf(pnorm, rnorm(10)))
  expect_no_error(validate_cdf(pbeta, rbeta(10, 5, 5), shape1 = 5, shape2 = 5))
  expect_no_error(validate_cdf(pcauchy, rcauchy(10)))
  expect_no_error(validate_cdf(pf, rf(10, 4, 4), df1 = 4, df2 = 4))
  expect_warning(validate_cdf(pf, rcauchy(10), df1 = 4, df2 = 4))
  expect_no_error(validate_cdf(punif, runif(10)))
  expect_no_error(validate_cdf(
    pweibull,
    rweibull(100, shape = 4, scale = 1),
    shape = 4,
    scale = 1
  ))
  expect_warning(validate_cdf(pgeom, rgeom(10, prob = .1), prob = .1))
  expect_warning(validate_cdf(ppois, rpois(10, lambda = 5), lambda = 5))
  expect_warning(validate_cdf(pgamma, rcauchy(100), shape = 4, rate = 4))
  expect_warning(validate_cdf(pbinom, rbinom(10, 1, .5), size = 1, prob = .5))
})

##### Test that X is correctly validated

test_that("validate support checks", {
  f <- f_dist(3, 3)
  nd <- normal_dist()
  ed <- exponential_dist()
  bd <- bernoulli_dist()
  gd <- geometric_dist()
  bind <- binomial_dist()
  betad <- beta_dist()
  lnd <- lognormal_dist()
  ld <- laplace_dist()
  ald <- asymmetric_laplace_dist()
  igd <- inverse_gaussian_dist()
  hd <- halfnormal_dist()
  csd <- chisq_dist()
  gamd <- gamma_dist()
  wd <- weibull_dist()
  cd <- cauchy_dist()
  ## sd <- stable_dist(0, 1, -1, .5)
  pd <- pareto_dist(1, 1)

  normalx <- rnorm(10)
  posx <- rexp(10)
  intx <- c(0, rpois(10, 1))
  negx <- -posx
  x01 <- rbinom(10, 1, .5)
  unifx <- rbeta(10, 6, 6)

  ## F
  expect_error(validate_x(normalx, ed))
  expect_no_error(validate_x(posx, ed))
  expect_error(validate_x(intx, ed))
  expect_error(validate_x(negx, ed))
  expect_error(validate_x(x01, ed))
  expect_no_error(validate_x(unifx, ed))
  ## Normal
  expect_no_error(validate_x(normalx, nd))
  expect_no_error(validate_x(posx, nd))
  expect_no_error(validate_x(intx, nd))
  expect_no_error(validate_x(negx, nd))
  expect_no_error(validate_x(x01, nd))
  expect_no_error(validate_x(unifx, nd))
  ## Exponential
  expect_error(validate_x(normalx, ed))
  expect_no_error(validate_x(posx, ed))
  expect_error(validate_x(intx, ed))
  expect_error(validate_x(negx, ed))
  expect_error(validate_x(x01, ed))
  expect_no_error(validate_x(unifx, ed))
  ## Bernoulli
  expect_error(validate_x(normalx, bd))
  expect_error(validate_x(posx, bd))
  expect_error(validate_x(intx, bd))
  expect_error(validate_x(negx, bd))
  expect_no_error(validate_x(x01, bd))
  expect_error(validate_x(unifx, bd))
  ##Geometric
  expect_error(validate_x(normalx, gd))
  expect_error(validate_x(posx, gd))
  expect_no_error(validate_x(intx, gd))
  expect_no_error(validate_x(intx[intx > 0], gd))
  expect_error(validate_x(negx, gd))
  expect_no_error(validate_x(x01, gd))
  expect_error(validate_x(unifx, gd))
  ## Binomial dist
  expect_error(validate_x(normalx, bind))
  expect_error(validate_x(posx, bind))
  expect_error(validate_x(intx, bind))
  expect_error(validate_x(negx, bind))
  expect_no_error(validate_x(x01, bind))
  expect_error(validate_x(unifx, bind))
  ## Beta Dist
  expect_error(validate_x(normalx, betad))
  expect_error(validate_x(posx, betad))
  expect_error(validate_x(intx, betad))
  expect_error(validate_x(negx, betad))
  expect_error(validate_x(x01, betad))
  expect_no_error(validate_x(unifx, betad))
  ## lognormal dist
  expect_error(validate_x(normalx, lnd))
  expect_no_error(validate_x(posx, lnd))
  expect_error(validate_x(intx, lnd))
  expect_error(validate_x(negx, lnd))
  expect_error(validate_x(x01, lnd))
  expect_no_error(validate_x(unifx, lnd))
  ## Laplace
  expect_no_error(validate_x(normalx, ld))
  expect_no_error(validate_x(posx, ld))
  expect_no_error(validate_x(intx, ld))
  expect_no_error(validate_x(negx, ld))
  expect_no_error(validate_x(x01, ld))
  expect_no_error(validate_x(unifx, ld))
  ## ALD
  expect_no_error(validate_x(normalx, ald))
  expect_no_error(validate_x(posx, ald))
  expect_no_error(validate_x(intx, ald))
  expect_no_error(validate_x(negx, ald))
  expect_no_error(validate_x(x01, ald))
  expect_no_error(validate_x(unifx, ald))
  ## IG
  expect_error(validate_x(normalx, igd))
  expect_no_error(validate_x(posx, igd))
  expect_error(validate_x(intx, igd))
  expect_error(validate_x(negx, igd))
  expect_error(validate_x(x01, igd))
  expect_no_error(validate_x(unifx, igd))
  ## Half Norm
  expect_error(validate_x(normalx, hd))
  expect_no_error(validate_x(posx, hd))
  expect_error(validate_x(intx, hd))
  expect_error(validate_x(negx, hd))
  expect_error(validate_x(x01, hd))
  expect_no_error(validate_x(unifx, hd))
  ## Chi Sq
  expect_error(validate_x(normalx, csd))
  expect_no_error(validate_x(posx, csd))
  expect_error(validate_x(intx, csd))
  expect_error(validate_x(negx, csd))
  expect_error(validate_x(x01, csd))
  expect_no_error(validate_x(unifx, csd))
  ## Gamma
  expect_error(validate_x(normalx, gamd))
  expect_no_error(validate_x(posx, gamd))
  expect_error(validate_x(intx, gamd))
  expect_error(validate_x(negx, gamd))
  expect_error(validate_x(x01, gamd))
  expect_no_error(validate_x(unifx, gamd))
  ## Weibull
  expect_error(validate_x(normalx, wd))
  expect_no_error(validate_x(posx, wd))
  expect_error(validate_x(intx, wd))
  expect_error(validate_x(negx, wd))
  expect_error(validate_x(x01, wd))
  expect_no_error(validate_x(unifx, wd))
  ## Cauchy
  expect_no_error(validate_x(normalx, cd))
  expect_no_error(validate_x(posx, cd))
  expect_no_error(validate_x(intx, cd))
  expect_no_error(validate_x(negx, cd))
  expect_no_error(validate_x(x01, cd))
  expect_no_error(validate_x(unifx, cd))
  ## Pareto
  expect_error(validate_x(normalx, pd))
  expect_error(validate_x(posx, pd))
  expect_error(validate_x(intx, pd))
  expect_error(validate_x(intx[intx > 0], pd))
  expect_error(validate_x(negx, pd))
  expect_error(validate_x(x01, pd))
  expect_error(validate_x(unifx, pd))
  ## Stable
  ## expect_error(validate_x(normalx, sd))
  ## expect_error(validate_x(posx, sd))
  ## expect_error(validate_x(intx, sd))
  ## expect_error(validate_x(intx[intx > 0], sd))
  ## expect_no_error(validate_x(negx, sd))
  ## expect_error(validate_x(x01, sd))
  ## expect_error(validate_x(unifx, sd))
})

#### Tests for expected distances

##### F

test_that("F EYY Distance", {
  d <- f_dist(10, 10)
  x <- d$sampler(1e6, d$par)
  y <- d$sampler(1e6, d$par)
  mc <- mean(abs(x - y))
  dd <- d$EYY(d$par)
  err <- (mc - dd) / dd
  expect_lt(err, 0.01)
})

test_that("F EXY Distance", {
  d <- f_dist(10, 10)
  y <- d$sampler(1e6, d$par)
  mc <- mean(abs(5 - y))
  dd <- d$EXYhat(5, d$par)
  err <- (mc - dd) / dd
  expect_lt(err, 0.01)
})


##### Normal

test_that("Normal EYY Distance", {
  d <- normal_dist(10, 10)
  x <- d$sampler(1e6, d$par)
  y <- d$sampler(1e6, d$par)
  mc <- mean(abs(x - y))
  dd <- d$EYY(d$par)
  err <- (mc - dd) / dd
  expect_lt(err, 0.01)
})

test_that("Normal EXY Distance", {
  d <- normal_dist(10, 10)
  y <- d$sampler(1e6, d$par)
  mc <- mean(abs(5 - y))
  dd <- d$EXYhat(5, d$par)
  err <- (mc - dd) / dd
  expect_lt(err, 0.01)
})


##### Uniform

test_that("Uniform Distances", {
  d <- uniform_dist(-10, 10)
  x <- d$sampler(1e6, d$par)
  y <- d$sampler(1e6, d$par)
  ## EYY
  mc1 <- mean(abs(x - y))
  dd1 <- d$EYY(d$par)
  err1 <- (mc1 - dd1) / dd1
  expect_lt(err1, 0.01)
  ## EXY
  mc2 <- mean(abs(4 - y))
  dd2 <- d$EXYhat(4, d$par)
  err2 <- (mc2 - dd2) / dd2
  expect_lt(err2, 0.01)
})

##### Exp
test_that("Exp Distances", {
  d <- exponential_dist(10)
  x <- d$sampler(1e6, d$par)
  y <- d$sampler(1e6, d$par)
  ## EYY
  mc1 <- mean(abs(x - y))
  dd1 <- d$EYY(d$par)
  err1 <- (mc1 - dd1) / dd1
  expect_lt(err1, 0.01)
  ## EXY
  mc2 <- mean(abs(4 - y))
  dd2 <- d$EXYhat(4, d$par)
  err2 <- (mc2 - dd2) / dd2
  expect_lt(err2, 0.01)
})

##### Poisson
test_that("Poisson Distances", {
  d <- poisson_dist(10)
  x <- d$sampler(1e6, d$par)
  y <- d$sampler(1e6, d$par)
  ## EYY
  mc1 <- mean(abs(x - y))
  dd1 <- d$EYY(d$par)
  err1 <- (mc1 - dd1) / dd1
  expect_lt(err1, 0.01)
  ## EXY
  mc2 <- mean(abs(4 - y))
  dd2 <- d$EXYhat(4, d$par)
  err2 <- (mc2 - dd2) / dd2
  expect_lt(err2, 0.01)
})

##### Bernoulli
test_that("Bernoulli Distances", {
  d <- bernoulli_dist(.7)
  x <- d$sampler(1e6, d$par)
  y <- d$sampler(1e6, d$par)
  ## EYY
  mc1 <- mean(abs(x - y))
  dd1 <- d$EYY(d$par)
  err1 <- (mc1 - dd1) / dd1
  expect_lt(err1, 0.01)
  ## EXY
  mc2 <- mean(abs(.3 - y))
  dd2 <- d$EXYhat(.3, d$par)
  err2 <- (mc2 - dd2) / dd2
  expect_lt(err2, 0.01)
})

##### Binomial
test_that("Binomial Distances", {
  d <- binomial_dist(10, .4)
  x <- d$sampler(1e6, d$par)
  y <- d$sampler(1e6, d$par)
  ## EYY
  mc1 <- mean(abs(x - y))
  dd1 <- d$EYY(d$par)
  err1 <- (mc1 - dd1) / dd1
  expect_lt(err1, 0.01)
  ## EXY
  mc2 <- mean(abs(4 - y))
  dd2 <- d$EXYhat(4, d$par)
  err2 <- (mc2 - dd2) / dd2
  expect_lt(err2, 0.01)
})

##### Beta
test_that("Beta Distances", {
  d <- beta_dist(10, 5)
  x <- d$sampler(1e6, d$par)
  y <- d$sampler(1e6, d$par)
  ## EYY
  mc1 <- mean(abs(x - y))
  dd1 <- d$EYY(d$par)
  err1 <- (mc1 - dd1) / dd1
  expect_lt(err1, 0.01)
  ## EXY
  mc2 <- mean(abs(.4 - y))
  dd2 <- d$EXYhat(.4, d$par)
  err2 <- (mc2 - dd2) / dd2
  expect_lt(err2, 0.01)
})

##### Geo
test_that("Geometric Distances", {
  d <- geometric_dist(.07)
  x <- d$sampler(1e6, d$par)
  y <- d$sampler(1e6, d$par)
  ## EYY
  mc1 <- mean(abs(x - y))
  dd1 <- d$EYY(d$par)
  err1 <- (mc1 - dd1) / dd1
  expect_lt(err1, 0.01)
  ## EXY
  mc2 <- mean(abs(4 - y))
  dd2 <- d$EXYhat(4, d$par)
  err2 <- (mc2 - dd2) / dd2
  expect_lt(err2, 0.01)
})


##### Half-Normal

test_that("hn Distances", {
  d <- halfnormal_dist(7)
  x <- d$sampler(1e6, d$par)
  y <- d$sampler(1e6, d$par)
  ## EYY
  mc1 <- mean(abs(x - y))
  dd1 <- d$EYY(d$par)
  err1 <- (mc1 - dd1) / dd1
  expect_lt(err1, 0.01)
  ## EXY
  mc2 <- mean(abs(4 - y))
  dd2 <- d$EXYhat(4, d$par)
  err2 <- (mc2 - dd2) / dd2
  expect_lt(err2, 0.01)
})
##### Laplace

test_that("Laplace Distances", {
  d <- laplace_dist(4, 2)
  x <- d$sampler(1e6, d$par)
  y <- d$sampler(1e6, d$par)
  ## EYY
  mc1 <- mean(abs(x - y))
  dd1 <- d$EYY(d$par)
  err1 <- (mc1 - dd1) / dd1
  expect_lt(err1, 0.01)
  ## EXY
  mc2 <- mean(abs(8 - y))
  dd2 <- d$EXYhat(8, d$par)
  err2 <- (mc2 - dd2) / dd2
  expect_lt(err2, 0.01)
})
##### Lognormal

test_that("Lognormal Distances", {
  d <- lognormal_dist(4, 2)
  x <- d$sampler(1e6, d$par)
  y <- d$sampler(1e6, d$par)
  ## EYY
  mc1 <- mean(abs(x - y))
  dd1 <- d$EYY(d$par)
  err1 <- (mc1 - dd1) / dd1
  expect_lt(err1, 0.01)
  ## EXY
  mc2 <- mean(abs(1 - y))
  dd2 <- d$EXYhat(1, d$par)
  err2 <- (mc2 - dd2) / dd2
  expect_lt(err2, 0.01)
})

##### A-Laplace
test_that("Asymmetric Laplace Distances", {
  d <- alaplace_dist(4, 3, .5)
  x <- d$sampler(1e6, d$par)
  y <- d$sampler(1e6, d$par)
  ## EYY
  mc1 <- mean(abs(x - y))
  dd1 <- d$EYY(d$par)
  err1 <- (mc1 - dd1) / dd1
  expect_lt(err1, 0.01)
  ## EXY
  mc2 <- mean(abs(8 - y))
  dd2 <- d$EXYhat(8, d$par)
  err2 <- (mc2 - dd2) / dd2
  expect_lt(err2, 0.01)
})


##### Weibull
test_that("Weibull Distances", {
  d <- weibull_dist(5, 6)
  x <- d$sampler(1e6, d$par)
  y <- d$sampler(1e6, d$par)
  ## EYY
  mc1 <- mean(abs(x - y))
  dd1 <- d$EYY(d$par)
  err1 <- (mc1 - dd1) / dd1
  expect_lt(err1, 0.01)
  ## EXY
  mc2 <- mean(abs(4 - y))
  dd2 <- d$EXYhat(4, d$par)
  err2 <- (mc2 - dd2) / dd2
  expect_lt(err2, 0.01)
})

##### Gamma
test_that("Gamma Distances", {
  d <- gamma_dist(5, 8)
  x <- d$sampler(1e6, d$par)
  y <- d$sampler(1e6, d$par)
  ## EYY
  mc1 <- mean(abs(x - y))
  dd1 <- d$EYY(d$par)
  err1 <- (mc1 - dd1) / dd1
  expect_lt(err1, 0.01)
  ## EXY
  mc2 <- mean(abs(4 - y))
  dd2 <- d$EXYhat(4, d$par)
  err2 <- (mc2 - dd2) / dd2
  expect_lt(err2, 0.01)
})

##### Chi-Sq
test_that("Chi-sq Distances", {
  d <- chisq_dist(33)
  x <- d$sampler(1e6, d$par)
  y <- d$sampler(1e6, d$par)
  ## EYY
  mc1 <- mean(abs(x - y))
  dd1 <- d$EYY(d$par)
  err1 <- (mc1 - dd1) / dd1
  expect_lt(err1, 0.01)
  ## EXY
  mc2 <- mean(abs(50 - y))
  dd2 <- d$EXYhat(50, d$par)
  err2 <- (mc2 - dd2) / dd2
  expect_lt(err2, 0.01)
})

##### Inv Gaussian
test_that("Inv Gaussian Distances", {
  d <- invgauss_dist(10, 10)
  x <- d$sampler(1e6, d$par)
  y <- d$sampler(1e6, d$par)
  ## EYY
  mc1 <- mean(abs(x - y))
  dd1 <- d$EYY(d$par)
  err1 <- (mc1 - dd1) / dd1
  expect_lt(err1, 0.01)
  ## EXY
  mc2 <- mean(abs(5 - y))
  dd2 <- d$EXYhat(5, d$par)
  err2 <- (mc2 - dd2) / dd2
  expect_lt(err2, 0.01)
})


##### Pareto
test_that("Pareto Distances", {
  d <- pareto_dist(4, 4)
  x <- d$sampler(1e6, d$par)
  y <- d$sampler(1e6, d$par)
  ## EYY
  mc1 <- mean(abs(x - y))
  dd1 <- d$EYY(d$par)
  err1 <- (mc1 - dd1) / dd1
  expect_lt(err1, 0.01)
  ## EXY
  mc2 <- mean(abs(5 - y))
  dd2 <- d$EXYhat(5, d$par)
  err2 <- (mc2 - dd2) / dd2
  expect_lt(err2, 0.01)
})

##### Cauchy
test_that("Cauchy Distances", {
  d <- cauchy_dist(2, .1, .5)
  x <- d$sampler(1e7, d$par)
  y <- d$sampler(1e7, d$par)
  ## EYY
  mc1 <- mean(abs(x - y)^.5)
  dd1 <- d$EYY(d$par)
  err1 <- (mc1 - dd1) / dd1
  expect_lt(err1, 0.01)
  ## EXY
  mc2 <- mean(abs(1 - y)^.5)
  dd2 <- d$EXYhat(1, d$par)
  err2 <- (mc2 - dd2) / dd2
  expect_lt(err2, 0.01)
})


##### Stable

##### F Tests
test_that("Test should work", {
  x <- rf(1000, 10, 10)
  d <- f_dist(10, 10)
  o <- egofd(x, d, nsim = 60)
  o
  expect_gt(o$statistic, 0)
  expect_gt(o$p.value, 0.01)
})

test_that("Test should work", {
  x <- rf(1000, 50, 50)
  d <- f_dist(50, 50)
  o <- egofd(x, d, nsim = 60)
  o
  expect_gt(o$statistic, 0)
  expect_gt(o$p.value, 0.01)
})

test_that("Test should detect weibull", {
  x <- rweibull(1000, 4, 4)
  d <- f_dist(4, 4)
  o <- egofd(x, d, nsim = 60)
  o
  expect_gt(o$statistic, 0)
  expect_lt(o$p.value, 0.01)
})

##### Normal Tests
test_that("Normal should not be transformed", {
  x <- rnorm(10)
  o <- egofd(x, normal_dist(0, 1), 0)
  expect_identical(names(o$statistic), "E-statistic")
})

test_that("egofd should return htest, even when nsim is zero", {
  x <- rnorm(10)
  o <- egofd(x, normal_dist(0, 1), 0)
  expect_s3_class(o, "htest")
  expect_gt(o$statistic, 0)
})

test_that("Normal p-vals should be uniform under Null hypothesis", {
  n <- 25
  save <- numeric(n)
  for (i in 1:n) {
    x <- rnorm(100, 0, 1)
    o <- egofd(x, normal_dist(0, 1), 100)
    save[i] <- unlist(o$p.value)
  }
  expect_gt(egofd(save, uniform_dist(0, 1), 100)$p.value, 0.05)
  expect_gt(o$statistic, 0)
})

test_that("Power to detect mean ", {
  x <- rnorm(100, 1, 1)
  o <- egofd(x, normal_dist(0, 1), 100)
  expect_lt(o$p.value, 0.05)
  expect_gt(o$statistic, 0)
})

test_that("Power to detect sd shift.", {
  x <- rnorm(100, 0, 3)
  o <- egofd(x, normal_dist(0, 1), 100)
  expect_lt(o$p.value, 0.05)
  expect_gt(o$statistic, 0)
})

test_that("Composite Test should work", {
  x <- rnorm(10)
  o <- egofd(x, normal_dist(), 0)
  expect_gt(o$statistic, 0)
})

##### Uniform tests

test_that("Test should work", {
  x <- runif(100)
  d <- uniform_dist(0, 1)
  o <- egofd(x, d, nsim = 60)
  o
  expect_gt(o$statistic, 0)
  expect_gt(o$p.value, 0.01)
})

test_that("Test should work", {
  x <- runif(100, -10, 10)
  d <- uniform_dist(-10, 10)
  o <- egofd(x, d, nsim = 60)
  o
  expect_gt(o$statistic, 0)
  expect_gt(o$statistic, 0.01)
})

test_that("Detect Beta", {
  x <- rbeta(100, 20, 20)
  d <- uniform_dist(0, 1)
  o <- egofd(x, d, nsim = 60)
  o
  expect_gt(o$statistic, 0)
  expect_lt(o$p.value, .1)
})

##### Exponential tests

test_that("Test should work", {
  x <- rexp(1000)
  d <- exponential_dist(1)
  o <- egofd(x, d, nsim = 60)
  o
  expect_gt(o$statistic, 0)
  expect_gt(o$p.value, 0.01)
})

test_that("Test should work", {
  x <- rexp(1000, 50)
  d <- exponential_dist(50)
  o <- egofd(x, d, nsim = 60)
  o
  expect_gt(o$statistic, 0)
  expect_gt(o$p.value, 0.01)
})

test_that("Test should detect weibull", {
  x <- rweibull(1000, 4, 4)
  d <- exponential_dist(4)
  o <- egofd(x, d, nsim = 60)
  o
  expect_gt(o$statistic, 0)
  expect_lt(o$p.value, 0.01)
})

test_that("Composite should work", {
  x <- rexp(100, 4)
  d <- exponential_dist()
  o <- egofd(x, d, nsim = 60)
  o
  expect_gt(o$statistic, 0)
  expect_gt(o$p.value, 0.01)
})

test_that("Composite should work detect weibull", {
  x <- rweibull(100, 4, 4)
  d <- exponential_dist()
  o <- egofd(x, d, nsim = 60)
  o
  expect_gt(o$statistic, 0)
  expect_lt(o$p.value, 0.01)
})

##### Bernoulli tests
test_that("Test works", {
  ## sometimes negative??
  x <- rbinom(100, 1, .5)
  d <- bernoulli_dist(.5)
  o <- egofd(x, d, nsim = 60)
  o
  expect_gte(o$statistic, 0)
  expect_gt(o$p.value, 0.01)
})

test_that("Detect p shift", {
  x <- rbinom(100, 1, .8)
  d <- bernoulli_dist(.5)
  o <- egofd(x, d, nsim = 60)
  o
  expect_gt(o$statistic, 0)
  expect_lt(o$p.value, 0.01)
})

##### Binomial tests

test_that("Test works", {
  x <- rbinom(100, 10, .5)
  d <- binomial_dist(10, .5)
  o <- egofd(x, d, nsim = 60)
  o
  expect_gt(o$statistic, 0)
  expect_gt(o$p.value, 0.01)
})

test_that("Detect p shift", {
  x <- rbinom(100, 10, .8)
  d <- binomial_dist(10, .5)
  o <- egofd(x, d, nsim = 60)
  o
  expect_gt(o$statistic, 0)
  expect_lt(o$p.value, 0.01)
})

test_that("Detect size shift", {
  x <- rbinom(100, 5, .5)
  d <- binomial_dist(10, .5)
  o <- egofd(x, d, nsim = 60)
  o
  expect_gt(o$statistic, 0)
  expect_lt(o$p.value, 0.01)
})

##### Geometric tests
test_that("Test works", {
  x <- rgeom(100, .5)
  d <- geometric_dist(.5)
  o <- egofd(x, d, nsim = 60)
  o
  expect_gt(o$statistic, 0)
  expect_gt(o$p.value, 0.01)
})

test_that("Detect p shift", {
  x <- rgeom(100, .2)
  d <- geometric_dist(.5)
  o <- egofd(x, d, nsim = 60)
  o
  expect_gt(o$statistic, 0)
  expect_lt(o$p.value, 0.01)
})

##### Poisson tests
test_that("Test works", {
  x <- rpois(100, 10)
  d <- poisson_dist(10)
  o <- egofd(x, d, nsim = 60)
  o
  expect_gt(o$statistic, 0)
  expect_gt(o$p.value, 0.01)
})

test_that("Detect lam shift", {
  x <- rpois(100, 5)
  d <- poisson_dist(10)
  o <- egofd(x, d, nsim = 60)
  o
  expect_gt(o$statistic, 0)
  expect_lt(o$p.value, 0.01)
})

test_that("Composite Test works", {
  x <- rpois(100, 10)
  d <- poisson_dist()
  o <- egofd(x, d, nsim = 60)
  o
  expect_gt(o$statistic, 0)
  expect_gt(o$p.value, 0.01)
})


##### Asym Laplace tests

test_that("Test works", {
  d <- asymmetric_laplace_dist(3, 3, 1)
  x <- d$sampler(10, d$par)
  o <- egofd(x, d, nsim = 60)
  o
  expect_gt(o$statistic, 0)
  expect_gt(o$p.value, 0.01)
})

test_that("Test works", {
  set.seed(1)
  d <- asymmetric_laplace_dist(1, 10, 1)
  x <- d$sampler(10, d$par)
  o <- egofd(x, d, nsim = 60)
  o
  expect_gt(o$statistic, 0)
  expect_gt(o$p.value, 0.01)
})


test_that("Test works", {
  d <- asymmetric_laplace_dist(10, 10, 3)
  x <- d$sampler(10, d$par)
  o <- egofd(x, d, nsim = 60)
  o
  expect_gt(o$statistic, 0)
  expect_gt(o$p.value, 0.01)
})

test_that("Test works", {
  d <- asymmetric_laplace_dist(1, 10, 1 / 10)
  x <- d$sampler(10, d$par)
  o <- egofd(x, d, nsim = 60)
  o
  expect_gt(o$statistic, 0)
  expect_gt(o$p.value, 0.01)
})

test_that("Test is sensitive", {
  d <- asymmetric_laplace_dist(1, 1, 1 / 10)
  x <- rnorm(100)
  o <- egofd(x, d, nsim = 60)
  o
  expect_gt(o$statistic, 0)
  expect_lt(o$p.value, 0.01)
})

test_that("Test is sensitive", {
  d <- asymmetric_laplace_dist(1, 1, 1)
  x <- rnorm(100)
  o <- egofd(x, d, nsim = 60)
  o
  expect_gt(o$statistic, 0)
  expect_lt(o$p.value, 0.01)
})


test_that("Test is sensitive", {
  d <- asymmetric_laplace_dist(3, 1, 1)
  x <- rnorm(100)
  o <- egofd(x, d, nsim = 60)
  o
  expect_gt(o$statistic, 0)
  expect_lt(o$p.value, 0.01)
})

test_that("Test is sensitive", {
  d <- asymmetric_laplace_dist(3, 1, 10)
  x <- rnorm(100)
  o <- egofd(x, d, nsim = 60)
  o
  expect_gt(o$statistic, 0)
  expect_lt(o$p.value, 0.01)
})

test_that("Test is sensitive", {
  d <- asymmetric_laplace_dist(3, 1, 1 / 10)
  x <- rnorm(100)
  o <- egofd(x, d, nsim = 60)
  o
  expect_gt(o$statistic, 0)
  expect_lt(o$p.value, 0.01)
})

test_that("Composite Test Works", {
  d <- asymmetric_laplace_dist(0, 1, 2)
  x <- d$sampler(100, par = d$par)
  o <- egofd(x, d, nsim = 60)
  o
  expect_gt(o$statistic, 0)
  expect_gt(o$p.value, 0.01)
})

test_that("Composite Test Works", {
  d <- asymmetric_laplace_dist(0, 3, 1 / 2)
  x <- d$sampler(100, par = d$par)
  o <- egofd(x, d, nsim = 60)
  o
  expect_gt(o$statistic, 0)
  expect_gt(o$p.value, 0.01)
})

test_that("Composite Test is sensitive", {
  set.seed(1) # why is this needed?
  d <- asymmetric_laplace_dist()
  x <- rcauchy(1000)
  o <- egofd(x, d, nsim = 60)
  o
  expect_gt(o$statistic, 0)
  expect_lt(o$p.value, 0.01)
})

test_that("Composite Test is sensitive", {
  d <- asymmetric_laplace_dist()
  x <- rnorm(4000, 3, 3)
  o <- egofd(x, d, nsim = 60)
  o
  expect_gt(o$statistic, 0)
  expect_lt(o$p.value, 0.01)
})


##### Inv Gaussian

test_that("Test works", {
  d <- inverse_gaussian_dist(3, 2)
  x <- d$sampler(10, d$par)
  o <- egofd(x, d, nsim = 60)
  o
  expect_gt(o$statistic, 0)
  expect_gt(o$p.value, 0.01)
})

test_that("Test works", {
  d <- inverse_gaussian_dist(1, 1)
  x <- d$sampler(10, d$par)
  o <- egofd(x, d, nsim = 60)
  o
  expect_gt(o$statistic, 0)
  expect_gt(o$p.value, 0.01)
})


test_that("Test works", {
  d <- inverse_gaussian_dist(5, 5)
  x <- d$sampler(10, d$par)
  o <- egofd(x, d, nsim = 50)
  o
  expect_gt(o$statistic, 0)
  expect_gt(o$p.value, 0.01)
})

test_that("Test works", {
  set.seed(1)
  d <- inverse_gaussian_dist(.1, .1)
  x <- d$sampler(10, d$par)
  o <- egofd(x, d, nsim = 50)
  o
  expect_gt(o$statistic, 0)
  expect_gt(o$p.value, 0.01)
})


test_that("Test works", {
  d <- inverse_gaussian_dist(.1, 10)
  x <- d$sampler(10, d$par)
  o <- egofd(x, d, nsim = 50)
  o
  expect_gt(o$statistic, 0)
  expect_gt(o$p.value, 0.01)
})


test_that("Test works", {
  d <- inverse_gaussian_dist(10, .1)
  x <- d$sampler(10, d$par)
  o <- egofd(x, d, nsim = 50)
  o
  expect_gt(o$statistic, 0)
  expect_gt(o$p.value, 0.01)
})


test_that("Composite Test works", {
  set.seed(1)
  d <- inverse_gaussian_dist()
  x <- statmod::rinvgauss(10, 5, 5)
  o <- egofd(x, d, nsim = 50)
  o
  expect_gt(o$statistic, 0)
  expect_gt(o$p.value, 0.01)
})

test_that("Composite Test works", {
  d <- inverse_gaussian_dist()
  x <- statmod::rinvgauss(1000, 1, 1)
  o <- egofd(x, d, nsim = 50)
  o
  expect_gt(o$statistic, 0)
  expect_gt(o$p.value, 0.01)
})


test_that("Composite Test is sensitive", {
  d <- inverse_gaussian_dist()
  x <- rweibull(1000, 4, 5)
  o <- egofd(x, d, nsim = 50)
  o
  expect_gt(o$statistic, 0)
  expect_lt(o$p.value, 0.01)
})

test_that("Simple test is sensitive ", {
  d <- inverse_gaussian_dist(3, 1)
  x <- statmod::rinvgauss(100, 3, 3)
  o <- egofd(x, d, nsim = 50)
  o
  expect_gt(o$statistic, 0)
  expect_lt(o$p.value, 0.05)
})

test_that("Simple test is sensitive ", {
  d <- inverse_gaussian_dist(3, 1)
  x <- rweibull(100, 3, 3)
  o <- egofd(x, d, nsim = 50)
  o
  expect_gt(o$statistic, 0)
  expect_lt(o$p.value, 0.01)
})
##### Half Norm tests

test_that("Test works", {
  d <- halfnormal_dist(1)
  x <- d$sampler(100, d$par)
  o <- egofd(x, d, nsim = 50)
  o
  expect_gt(o$statistic, 0)
  expect_gt(o$p.value, 0.01)
})

test_that("Test works", {
  d <- halfnormal_dist(5)
  x <- d$sampler(100, d$par)
  o <- egofd(x, d, nsim = 50)
  o
  expect_gt(o$statistic, 0)
  expect_gt(o$p.value, 0.01)
})

test_that("Test works", {
  d <- halfnormal_dist(50)
  x <- d$sampler(100, d$par)
  o <- egofd(x, d, nsim = 50)
  o
  expect_gt(o$statistic, 0)
  expect_gt(o$p.value, 0.01)
})

test_that("Test is sensitive", {
  d <- halfnormal_dist(1)
  x <- rweibull(100, 3, 3)
  o <- egofd(x, d, nsim = 50)
  o
  expect_gt(o$statistic, 0)
  expect_lt(o$p.value, 0.01)
})

test_that("Test is sensitive", {
  d <- halfnormal_dist(1)
  x <- rexp(1000)
  o <- egofd(x, d, nsim = 50)
  o
  expect_gt(o$statistic, 0)
  expect_lt(o$p.value, 0.01)
})

test_that("Test is sensitive", {
  d <- halfnormal_dist(scale = 1)
  x <- rexp(1000)
  o <- egofd(x, d, nsim = 50)
  o
  expect_gt(o$statistic, 0)
  expect_lt(o$p.value, 0.01)
})

test_that("Composite test works", {
  d <- halfnormal_dist(scale = 1)
  x <- d$sampler(1000, d$par)
  o <- egofd(x, halfnormal_dist(), nsim = 50)
  o
  expect_gt(o$statistic, 0)
  expect_gt(o$p.value, 0.01)
})


test_that("Composite test is sensitive", {
  x <- rexp(1000)
  o <- egofd(x, halfnormal_dist(), nsim = 50)
  o
  expect_gt(o$statistic, 0)
  expect_lt(o$p.value, 0.01)
})

test_that("Composite test is sensitive", {
  x <- rweibull(1000, 3, 3)
  o <- egofd(x, halfnormal_dist(), nsim = 50)
  o
  expect_gt(o$statistic, 0)
  expect_lt(o$p.value, 0.01)
})

##### Chi Sq tests
test_that("Test works", {
  d <- chisq_dist(5)
  x <- d$sampler(100, d$par)
  o <- egofd(x, d, nsim = 50)
  o
  expect_gt(o$statistic, 0)
  expect_gt(o$p.value, 0.01)
})

test_that("Test works", {
  set.seed(1)
  d <- chisq_dist(1)
  x <- d$sampler(1000, d$par)
  o <- egofd(x, d, nsim = 50)
  o
  expect_gt(o$statistic, 0)
  expect_gt(o$p.value, 0.01)
})

test_that("Test works", {
  d <- chisq_dist(40)
  x <- d$sampler(100, d$par)
  o <- egofd(x, d, nsim = 50)
  o
  expect_gt(o$statistic, 0)
  expect_gt(o$p.value, 0.01)
})


test_that("Test is sensitive", {
  d <- chisq_dist(40)
  x <- rchisq(1000, 4)
  o <- egofd(x, d, nsim = 50)
  o
  expect_gt(o$statistic, 0)
  expect_lt(o$p.value, 0.01)
})

test_that("Test is sensitive", {
  d <- chisq_dist(40)
  x <- rexp(1000, 4)
  o <- egofd(x, d, nsim = 50)
  o
  expect_gt(o$statistic, 0)
  expect_lt(o$p.value, 0.01)
})


##### Gamma  tests

test_that("Test works", {
  d <- gamma_dist(1, 1)
  x <- d$sampler(100, d$par)
  o <- egofd(x, d, nsim = 50)
  o
  expect_gt(o$statistic, 0)
  expect_gt(o$p.value, 0.01)
})

test_that("Test works", {
  d <- gamma_dist(3, 3)
  x <- d$sampler(100, d$par)
  o <- egofd(x, d, nsim = 50)
  o
  expect_gt(o$statistic, 0)
  expect_gt(o$p.value, 0.01)
})

test_that("Test works", {
  d <- gamma_dist(5, 2)
  x <- d$sampler(1000, d$par)
  o <- egofd(x, d, nsim = 50)
  o
  expect_gt(o$statistic, 0)
  expect_gt(o$p.value, 0.01)
})

test_that("Test works", {
  d <- gamma_dist(2, 5)
  x <- d$sampler(100, d$par)
  o <- egofd(x, d, nsim = 50)
  o
  expect_gt(o$statistic, 0)
  expect_gt(o$p.value, 0.01)
})

test_that("Test is sensitive", {
  d <- gamma_dist(2, 5)
  x <- rweibull(1000, 3, 3)
  o <- egofd(x, d, nsim = 50)
  o
  expect_gt(o$statistic, 0)
  expect_lt(o$p.value, 0.01)
})


##### Weibull  tests
test_that("Test works", {
  d <- weibull_dist(1, 1)
  x <- d$sampler(1000, d$par)
  o <- egofd(x, d, nsim = 50)
  o
  expect_gt(o$statistic, 0)
  expect_gt(o$p.value, 0.01)
})

test_that("Test works", {
  d <- weibull_dist(3, 3)
  x <- d$sampler(1000, d$par)
  o <- egofd(x, d, nsim = 50)
  o
  expect_gt(o$statistic, 0)
  expect_gt(o$p.value, 0.01)
})

test_that("Test works", {
  d <- weibull_dist(10, 1)
  x <- d$sampler(1000, d$par)
  o <- egofd(x, d, nsim = 50)
  o
  expect_gt(o$statistic, 0)
  expect_gt(o$p.value, 0.01)
})

test_that("Test works", {
  d <- weibull_dist(1, 10)
  x <- d$sampler(1000, d$par)
  o <- egofd(x, d, nsim = 50)
  o
  expect_gt(o$statistic, 0)
  expect_gt(o$p.value, 0.01)
})

test_that("Test is sensitive", {
  d <- weibull_dist(1, 10)
  x <- rgamma(1000, 3, 3)
  o <- egofd(x, d, nsim = 50)
  o
  expect_gt(o$statistic, 0)
  expect_lt(o$p.value, 0.01)
})

test_that("Composite test works", {
  d <- weibull_dist()
  x <- rweibull(1000, 3, 3)
  o <- egofd(x, d, nsim = 50)
  o
  expect_gt(o$statistic, 0)
  expect_gt(o$p.value, 0.01)
})

test_that("Composite test works", {
  d <- weibull_dist()
  x <- rweibull(1000, 10, 1)
  o <- egofd(x, d, nsim = 50)
  o
  expect_gt(o$statistic, 0)
  expect_gt(o$p.value, 0.01)
})


test_that("Composite test is sensitive", {
  d <- weibull_dist()
  x <- rgamma(1000, 10, 1)
  o <- egofd(x, d, nsim = 50)
  o
  expect_gt(o$statistic, 0)
  expect_lt(o$p.value, 0.01)
})
##### Cauchy tests

test_that("test works", {
  d <- cauchy_dist(10, 1)
  x <- rcauchy(1000, 10, 1)
  o <- egofd(x, d, nsim = 50)
  o
  expect_gt(o$statistic, 0)
  expect_gt(o$p.value, 0.01)
})

test_that("test works", {
  d <- cauchy_dist(10, 10)
  x <- rcauchy(1000, 10, 10)
  o <- egofd(x, d, nsim = 50)
  o
  expect_gt(o$statistic, 0)
  expect_gt(o$p.value, 0.01)
})

test_that("test is sensitive", {
  d <- cauchy_dist(10, 10)
  x <- rnorm(1000, 10, 10)
  o <- egofd(x, d, nsim = 50)
  o
  expect_gt(o$statistic, 0)
  expect_lt(o$p.value, 0.01)
})

test_that("composite test works", {
  d <- cauchy_dist()
  x <- rcauchy(1000, 10, 10)
  o <- egofd(x, d, nsim = 50)
  o
  expect_gt(o$statistic, 0)
  expect_gt(o$p.value, 0.01)
})

test_that("composite test is sensitive", {
  d <- cauchy_dist()
  x <- rnorm(1000, 10, 10)
  o <- egofd(x, d, nsim = 50)
  o
  expect_gt(o$statistic, 0)
  expect_lt(o$p.value, 0.01)
})

##### Stable tests

###### Symmetric Stable (beta = skew = 0)
## test_that("Test works", {
##   d <- stable_dist(0, 1, 0, 1)
##   x <- d$sampler(100, d$par)
##   o <- egofd(x, d)
##   o
##   expect_gt(o$statistic, 0)
##   expect_gt(o$p.value, 0.01)})
##
## test_that("Test works", {
##   d <- stable_dist(0, 1, 0, 1.5, 1 / 100)
##   x <- d$sampler(100, d$par)
##   o <- egofd(x, d)
##   o
##   expect_gt(o$statistic, 0)
##   expect_gt(o$p.value, 0.01)})
##
## test_that("Test works", {
##   d <- stable_dist(0, 1, 0, 2)
##   x <- d$sampler(100, d$par)
##   o <- egofd(x, d)
##   o
##   expect_gt(o$statistic, 0)
##   expect_gt(o$p.value, 0.01)})

###### Asymmetric Case

## test_that("Test works", {
##   d <- stable_dist(0, 1, .5, 1)
##   x <- d$sampler(100, d$par)
##   o <- egofd(x, d)
##   o
##   expect_gt(o$statistic, 0)
##   expect_gt(o$p.value, 0.01)})
##
##
## test_that("Test works", {
##   d <- stable_dist(0, 1, -.5, 1)
##   x <- d$sampler(100, d$par)
##   o <- egofd(x, d)
##   o
##   expect_gt(o$statistic, 0)
##   expect_gt(o$p.value, 0.01)})
##
## test_that("Test works", {
##   d <- stable_dist(0, 1, 1, 1, 0.25)
##   x <- d$sampler(100, d$par)
##   o <- egofd(x, d)
##   o
##   expect_gt(o$statistic, 0)
##   expect_gt(o$p.value, 0.01)})

##### Beta Test

test_that("Beta Estat should be positive", {
  d <- beta_dist(20, 20)
  x <- rbeta(100, 20, 20)
  o <- egofd(x, d, 0)
  o
  expect_gt(o$statistic, 0)
})

test_that("Beta Estat should be positive", {
  d <- beta_dist(.5, 1.5)
  x <- rbeta(100, .5, 1.5)
  o <- egofd(x, d, 0)
  o
  expect_gt(o$statistic, 0)
})

test_that("Beta Estat should be sensitive to parameter change", {
  d <- beta_dist(.5, 1.5)
  x <- rbeta(100, 20, 20)
  o <- egofd(x, d, 100)
  o
  expect_lt(o$p.value, .01)
})


test_that("Beta Composite should work", {
  d <- beta_dist()
  x <- rbeta(100, 20, 20)
  o <- egofd(x, d, 0)
  o
  expect_gt(o$statistic, 0)
})

test_that("Beta Composite should work", {
  d <- beta_dist()
  x <- rbeta(100, 1 / 60, 70)
  o <- egofd(x, d, 0)
  o
  expect_gt(o$statistic, 0)
})

##### Pareto Test
test_that("Pareto: shape, scale >1", {
  # erratic
  d <- pareto_dist(3, 3)
  x <- d$sampler(100, d$par)
  o <- egofd(x, d, nsim = 25)
  o
  expect_s3_class(o, "htest")
  expect_gt(o$statistic, 0)
})

test_that("Pareto: shape = scale = pow > 1", {
  # erratic
  expect_error(d <- pareto_dist(4, 4, 4))
})

test_that("Pareto: shape, scale >1", {
  d <- pareto_dist(5, 5)
  x <- d$sampler(100, d$par)
  o <- egofd(x, d, nsim = 100)
  o
  expect_s3_class(o, "htest")
  expect_gt(o$statistic, 0)
})


test_that("Pareto: shape, scale <1", {
  d <- pareto_dist(.1, .1)
  x <- d$sampler(100, d$par)
  o <- egofd(x, d, nsim = 0)
  o
  expect_s3_class(o, "htest")
  expect_gt(o$statistic, 0)
})

test_that("Pareto: shape, scale = 1", {
  d <- pareto_dist(1, 1)
  x <- d$sampler(100, d$par)
  o <- egofd(x, d, nsim = 0)
  o
  expect_s3_class(o, "htest")
  expect_gt(o$statistic, 0)
})

test_that("Pareto: mixed", {
  d <- pareto_dist(.1, 10)
  x <- d$sampler(100, d$par)
  o <- egofd(x, d, nsim = 0)
  o
  expect_s3_class(o, "htest")
  expect_gt(o$statistic, 0)
})

test_that("Pareto: mixed", {
  d <- pareto_dist(10, .1)
  x <- d$sampler(100, d$par)
  o <- egofd(x, d, nsim = 0)
  o
  expect_s3_class(o, "htest")
  expect_gt(o$statistic, 0)
})


test_that("Pareto: pow > shape", {
  expect_error(pareto_dist(10, .1, pow = 5))
})


test_that("Pareto: pow > shape", {
  expect_error(pareto_dist(10, .1, 1))
})

##### Binomial tests

test_that("Binomial", {
  x <- rbinom(10, 10, .5)
  d <- binomial_dist(size = 10, prob = .5)
  o <- egofd(x, d, 0)
  expect_s3_class(o, "htest")
  expect_gt(o$statistic, 0)
  x <- rbinom(10, 10, .9)
  o <- egofd(x, d, 99)
  expect_gt(o$statistic, 0)
  expect_lt(o$p.value, 0.5)
  x <- rexp(10)
  expect_error(binomial_dist(x))
})


##### Lognormal Tests

test_that("lnorm test", {
  x <- rlnorm(1000)
  o <- egofd(x, lognormal_dist(0, 1), nsim = 25)
  o
  expect_gt(o$statistic, 0)
})

test_that("lnorm test", {
  set.seed(1)
  x <- rlnorm(1000, 5, 5)
  d <- lognormal_dist(5, 5)
  o <- egofd(x, d, nsim = 25)
  o
  expect_gt(o$statistic, 0)
  expect_gt(o$p.value, 0.001)
})

test_that("lnorm test should detect meanlog shift", {
  x <- rlnorm(1000, 7, 5)
  d <- lognormal_dist(5, 5)
  o <- egofd(x, d, nsim = 25)
  o
  expect_gt(o$statistic, 0)
  expect_lt(o$p.value, 0.5)
})

test_that("lnorm composite test ", {
  x <- rlnorm(1000, 2, 2)
  d <- lognormal_dist()
  o <- egofd(x, d, nsim = 25)
  o
  expect_gt(o$statistic, 0)
  expect_gt(o$p.value, 0.01)
})

test_that("lnorm composite test can detect weibull ", {
  x <- rweibull(1000, 2, 2)
  d <- lognormal_dist()
  o <- egofd(x, d, nsim = 25)
  o
  expect_gt(o$statistic, 0)
  expect_lt(o$p.value, 0.05)
})

##### Laplace Tests

test_that("laplace test", {
  d <- laplace_dist(0, 1)
  x <- d$sampler(1000, d$par)
  o <- egofd(x, d, nsim = 25)
  o
  expect_gt(o$statistic, 0)
})

test_that("laplace test, different params. ", {
  d <- laplace_dist(5, 5)
  x <- d$sampler(1000, d$par)
  o <- egofd(x, d, nsim = 25)
  o
  expect_gt(o$statistic, 0)
  expect_gt(o$p.value, 0.001)
})

test_that("laplace test should detect location shift", {
  d <- laplace_dist(5, 5)
  x <- d$sampler(1000, list(location = 0, scale = 5))
  o <- egofd(x, d, nsim = 25)
  o
  expect_gt(o$statistic, 0)
  expect_lt(o$p.value, 0.05)
})

test_that("laplace composite test ", {
  d <- laplace_dist()
  x <- d$sampler(1000, list(location = 0, scale = 5))
  o <- egofd(x, d, nsim = 25)
  o
  expect_gt(o$statistic, 0)
  expect_gt(o$p.value, 0.01)
})

test_that("laplace composite test can detect normal ", {
  d <- laplace_dist()
  x <- rnorm(1000)
  o <- egofd(x, d, nsim = 25)
  o
  expect_gt(o$statistic, 0)
  expect_lt(o$p.value, 0.01)
})

#####
