#' Calculate Multifunctionality Divergence (MFdiv)
#'
#' @description
#' Multifunctionality divergence (MFdiv) quantifies the degree of functional dispersion
#' within an ecosystem by measuring how ecosystem functions deviate from their weighted centroid.
#' MFdiv is calculated using a weighted mean absolute deviation approach.
#'
#' @param data A data frame or matrix where rows represent observations (e.g., sites, plots)
#'        and columns represent ecosystem functions. Function values should be normalized
#'        to a common scale (e.g., 0-1).
#' @param weights A numeric vector of weights for each function (column). If NULL,
#'        equal weights are assigned to all functions. Default is NULL.
#'
#' @details
#' To measure MFdiv quantitatively, we employ a weighted mean absolute deviation approach.
#'
#' The MFdiv is calculated as:
#' \deqn{MFdiv = 2 \times \frac{\sum_{i=1}^{n} w_i |f_i - \mu|}{\sum_{i=1}^{n} w_i}}{MFdiv = 2 * (sum of weighted absolute deviations)/(sum of weights)}
#'
#' where:
#' \itemize{
#'   \item \eqn{f_i} is the normalized performance level of function \eqn{i} (where \eqn{f_i > 0})
#'   \item \eqn{w_i} is the weight of function \eqn{i}
#'   \item \eqn{\mu} is the weighted mean (centroid) of all function values:
#' }
#'
#' \deqn{\mu = \frac{\sum_{i=1}^{n} w_i f_i}{\sum_{i=1}^{n} w_i}}{mu = (sum of weighted function values)/(sum of weights)}
#'
#' This metric quantifies the average weighted deviation of each function from the community's
#' functional centroid. The multiplication by 2 scales the values to approximately [0, 1], where:
#' \itemize{
#'   \item 0 indicates complete functional convergence (all functions perform identically)
#'   \item Higher values represent greater functional divergence (functions are more dispersed
#'         across the performance gradient)
#' }
#'
#' Only positive function values (\eqn{f_i > 0}) are included in the calculation. Functions
#' with zero or negative values are automatically excluded along with their corresponding weights.
#'
#' @return A data frame with one column (MFdiv) containing the multifunctionality divergence
#'         values for each observation (row) in the input data. Row names are preserved from
#'         the input data.
#'
#' @examples
#' # Example with sample data
#' data <- data.frame(
#'   func1 = c(0.5, 0.3, 0.2),
#'   func2 = c(0.2, 0.4, 0.4),
#'   func3 = c(0.3, 0.3, 0.4)
#' )
#' rownames(data) <- c("Site1", "Site2", "Site3")
#'
#' # Calculate MFdiv with equal weights
#' MFdiv(data)
#'
#' # Calculate MFreg with custom weights (emphasizing func2)
#' MFdiv(data, weights = c(1, 2, 1))
#'
#'
#' @export
MFdiv <- function(data, weights = NULL) {
  if (ncol(data) <= 2) {
    stop("The number of functions must be greater than 2!")
  }
  # If no weights are provided, create a weight vector with all 1's
  if (is.null(weights)) {
    weights <- rep(1, ncol(data))
  }
  if (length(weights) != ncol(data)) {
    stop("The length of the weight vector must be equal to the number of columns in the data frame")
  }
  mfdiv_values <- apply(data, 1, function(fi) {
    wi <- weights
    wi <- wi[fi > 0]
    fi <- fi[fi > 0]
    mu <- sum(wi * fi) / sum(wi)
    weighted_deviations <- wi * abs(fi - mu)
    sum(weighted_deviations) / sum(wi)
  })
  mfdiv_values <- mfdiv_values * 2
  result <- data.frame(MFdiv = mfdiv_values)
  rownames(result) <- rownames(data)
  return(result)
}
