#' Flag Top Anomalies Based on Score Threshold
#'
#' Categorizes records as anomalous or normal based on their anomaly scores,
#' using either a fixed threshold or a contamination rate.
#'
#' @importFrom stats quantile
#' @param data_with_scores A data frame containing an \code{anomaly_score} column,
#'   typically the output of \code{score_anomaly()}.
#' @param threshold Numeric value between 0 and 1. Records with anomaly_score
#'   >= threshold are flagged as anomalous. If NULL (default), uses the
#'   contamination rate from the \code{score_anomaly()} attributes.
#' @param contamination Numeric value between 0 and 1. If \code{threshold} is NULL,
#'   this proportion of records with the highest scores will be flagged.
#'   Default is 0.05 (5%).
#'
#' @return The input data frame with an additional \code{is_anomaly} logical column
#'   indicating whether each record is flagged as anomalous.
#'
#' @export
#'
#' @examples
#' \donttest{
#' data <- data.frame(
#'   patient_id = 1:50,
#'   age = rnorm(50, 50, 15),
#'   cost = rnorm(50, 10000, 5000)
#' )
#' scored_data <- score_anomaly(data)
#' flagged_data <- flag_top_anomalies(scored_data, contamination = 0.05)
#' }
flag_top_anomalies <- function(data_with_scores, threshold = NULL, contamination = 0.05) {
  
  # Validate inputs
  if (!is.data.frame(data_with_scores)) {
    stop("data_with_scores must be a data frame")
  }
  
  if (!"anomaly_score" %in% names(data_with_scores)) {
    stop("data_with_scores must contain an 'anomaly_score' column")
  }
  
  if (contamination <= 0 || contamination >= 1) {
    stop("contamination must be between 0 and 1")
  }
  
  if (is.null(threshold)) {
    attr_contamination <- attr(data_with_scores, "contamination")
    if (!is.null(attr_contamination)) {
      contamination <- attr_contamination
      if (contamination <= 0 || contamination >= 1) {
        stop("contamination must be between 0 and 1")
      }
    }
    
    threshold <- stats::quantile(data_with_scores$anomaly_score, 1 - contamination, na.rm = TRUE)
  } else {
    if (threshold < 0 || threshold > 1) {
      stop("threshold must be between 0 and 1")
    }
  }
  
  data_with_scores$is_anomaly <- data_with_scores$anomaly_score >= threshold
  attr(data_with_scores, "anomaly_threshold") <- threshold
  
  return(data_with_scores)
}

