% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/aoe_expand.R
\name{aoe_expand}
\alias{aoe_expand}
\title{Adaptive AoE Expansion to Capture Minimum Points}
\usage{
aoe_expand(
  points,
  support = NULL,
  min_points,
  max_area = 2,
  max_dist = NULL,
  method = c("buffer", "stamp"),
  reference = NULL,
  mask = NULL,
  coords = NULL
)
}
\arguments{
\item{points}{An \code{sf} object with POINT geometries.}

\item{support}{One of:
\itemize{
\item \code{sf} object with POLYGON/MULTIPOLYGON geometries
\item Country name or ISO code: \code{"France"}, \code{"FR"}, \code{"FRA"}
\item Vector of countries: \code{c("France", "Germany")}
\item Missing: auto-detects countries containing the points
}}

\item{min_points}{Minimum number of points to capture in the AoE.
The function finds the smallest scale that includes at least this many points.}

\item{max_area}{Maximum halo area as a proportion of the original support area.
Default is 2, meaning halo area cannot exceed twice the support area
(total AoE <= 3x original). Set to \code{Inf} to disable.}

\item{max_dist}{Maximum expansion distance in CRS units. For the buffer method,
this is the maximum buffer distance. For the stamp method, this is converted
to a maximum scale based on the support's characteristic radius.
Default is \code{NULL} (no distance cap).}

\item{method}{Method for computing the area of effect:
\itemize{
\item \code{"buffer"} (default): Uniform buffer around the support boundary.
Robust for any polygon shape. Buffer distance is calculated to achieve
the target halo area.
\item \code{"stamp"}: Scale vertices outward from the centroid (or reference point).
Preserves shape proportions but only guarantees containment for
star-shaped polygons. May leave small gaps for highly concave shapes.
}}

\item{reference}{Optional \code{sf} object with a single POINT geometry.

If \code{NULL} (default), the centroid of each support is used.
Only valid when \code{support} has a single row and \code{method = "stamp"}.}

\item{mask}{Optional mask for clipping the area of effect. Can be:
\itemize{
\item \code{sf} object with POLYGON or MULTIPOLYGON geometry
\item \code{"land"}: use the bundled global land mask to exclude sea areas
If provided, each area of effect is intersected with this mask.
}}

\item{coords}{Column names for coordinates when \code{points} is a data.frame,
e.g. \code{c("lon", "lat")}. If \code{NULL}, auto-detects common names.}
}
\value{
An \code{aoe_result} object (same as \code{\link[=aoe]{aoe()}}) with additional attributes:
\describe{
\item{target_reached}{Logical: was \code{min_points} achieved for all supports?
Use \code{attr(result, "expansion_info")} for per-support details.}
\item{expansion_info}{Data frame with per-support expansion details:
support_id, scale_used, points_captured, target_reached, cap_hit.}
}
}
\description{
Expands the area of effect just enough to capture at least \code{min_points},
subject to hard caps on expansion. This is useful when a fixed scale leaves
some supports with insufficient data for stable modelling.
}
\details{
Unlike \code{\link[=aoe]{aoe()}}, which applies consistent geometry across all supports,
\code{aoe_expand()} adapts the scale per-support based on local point density.
Use with caution: this can make AoEs incomparable across regions with
different point densities.
\subsection{Algorithm}{

For each support, binary search finds the minimum scale where point count >= min_points.
The search is bounded by:
\itemize{
\item Lower: scale = 0 (core only)
\item Upper: minimum of max_area cap and max_dist cap
}

If the caps prevent reaching min_points, a warning is issued and the result
uses the maximum allowed scale.
}

\subsection{Caps}{

Two caps ensure AoE doesn't expand unreasonably:

\strong{max_area} (relative): Limits halo area to \code{max_area} times the original.
The corresponding scale is \code{sqrt(1 + max_area) - 1}.
Default max_area = 2 means scale <= 0.732 (total area <= 3x).

\strong{max_dist} (absolute): Limits expansion distance in CRS units.
For buffer method, this is the buffer distance directly.
For stamp method, converted to scale via \code{max_dist / characteristic_radius}
where characteristic_radius = sqrt(area / pi).
}
}
\examples{
library(sf)

# Create a support with sparse points
support <- st_as_sf(
  data.frame(id = 1),
  geometry = st_sfc(st_polygon(list(
    cbind(c(0, 100, 100, 0, 0), c(0, 0, 100, 100, 0))
  ))),
  crs = 32631
)

# Points scattered around
set.seed(42)
pts <- st_as_sf(
  data.frame(id = 1:50),
  geometry = st_sfc(lapply(1:50, function(i) {
    st_point(c(runif(1, -50, 150), runif(1, -50, 150)))
  })),
  crs = 32631
)

# Expand until we have at least 20 points
result <- aoe_expand(pts, support, min_points = 20)

# Check expansion info
attr(result, "expansion_info")

}
\seealso{
\code{\link[=aoe]{aoe()}} for fixed-scale AoE computation
}
