% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/vc_softbart_regression.R
\name{vc_softbart_regression}
\alias{vc_softbart_regression}
\title{SoftBart Varying Coefficient Regression}
\usage{
vc_softbart_regression(
  formula,
  linear_var_name,
  data,
  test_data,
  num_tree = 20,
  k = 2,
  hypers_intercept = NULL,
  hypers_slope = NULL,
  opts = NULL,
  verbose = TRUE,
  warn = TRUE
)
}
\arguments{
\item{formula}{A model formula with a numeric variable on the left-hand-side and non-linear predictors on the right-hand-side.}

\item{linear_var_name}{A string containing the variable in the data that is to be treated linearly.}

\item{data}{A data frame consisting of the training data.}

\item{test_data}{A data frame consisting of the testing data.}

\item{num_tree}{The number of trees in the ensemble to use.}

\item{k}{Determines the standard deviation of the leaf node parameters, which
is given by \code{3 / k / sqrt(num_tree)} (intercept) and defaults to
\code{1/k/sqrt(num_tree)} (slope). This can be modified for the slope by
specifying your own hyperparameter.}

\item{hypers_intercept}{A list of hyperparameters constructed from the \code{Hypers()} function (\code{num_tree}, \code{k}, and \code{sigma_mu} are overridden by this function).}

\item{hypers_slope}{A list of hyperparameters constructed from the \code{Hypers()} function (\code{num_tree} is overridden by this function).}

\item{opts}{A list of options for running the chain constructed from the \code{Opts()} function (\code{update_sigma} is overridden by this function).}

\item{verbose}{If \code{TRUE}, progress of the chain will be printed to the console.}

\item{warn}{If \code{TRUE}, remind the user that they probably don't want the linear term to be included in the formula for the nonlinear part.}
}
\value{
Returns a list with the following components
\itemize{
  \item \code{sigma_mu_alpha}: samples of the standard deviation of the leaf node parameters for the intercept.
  \item \code{sigma_mu_beta}: samples of the standard deviation of the leaf node parameters for the slope.
  \item \code{sigma}: samples of the error standard deviation.
  \item \code{var_counts_alpha}: a matrix with a column for each predictor group containing the number of times each predictor is used in the ensemble at each iteration for the intercept.
  \item \code{var_counts_beta}: a matrix with a column for each predictor group containing the number of times each predictor is used in the ensemble at each iteration for the slope.
  \item \code{alpha_train}: samples of the nonparametric intercept evaluated on the training set.
  \item \code{alpha_test}: samples of the nonparametric intercept evaluated on the test set.
  \item \code{beta_train}: samples of the nonparametric slope evaluated on the training set.
  \item \code{beta_test}: samples of the nonparametric slope evaluated on the test set.
  \item \code{mu_train}: samples of the predictions evaluated on the training set.
  \item \code{mu_test}: samples of the predictions evaluated on the test set.
  \item \code{formula}: the formula specified by the user.
  \item \code{ecdfs}: empirical distribution functions, used by the \code{predict} function.
  \item \code{opts}: the options used when running the chain.
  \item \code{mu_Y, sd_Y}: used with the \code{predict} function to transform predictions.
  \item \code{alpha_forest}: a forest object for the intercept; see the \code{MakeForest} documentation for more details.
  \item \code{beta_forest}: a forest object for the slope; see the \code{MakeForest} documentation for more details.
}
}
\description{
Fits a semiparametric varying coefficient regression model with the
nonparametric slope and intercept \deqn{Y = \alpha(X) + Z \beta(X) +
\epsilon}{Y = alpha + Z * beta(X) + epsilon} using a soft BART model.
}
\examples{

## NOTE: SET NUMBER OF BURN IN AND SAMPLE ITERATIONS HIGHER IN PRACTICE

num_burn <- 10 ## Should be ~ 5000
num_save <- 10 ## Should be ~ 5000

set.seed(1234)
f_fried <- function(x) 10 * sin(pi * x[,1] * x[,2]) + 20 * (x[,3] - 0.5)^2 +
  10 * x[,4] + 5 * x[,5]

gen_data <- function(n_train, n_test, P, sigma) {
  X <- matrix(runif(n_train * P), nrow = n_train)
  Z <- rnorm(n_train)
  r <- f_fried(X)
  mu <- Z * r
  X_test <- matrix(runif(n_test * P), nrow = n_test)
  Z_test <- rnorm(n_test)
  r_test <- f_fried(X_test)
  mu_test <- Z_test * r_test
  Y <- mu + sigma * rnorm(n_train)
  Y_test <- mu + sigma * rnorm(n_test)

  return(list(X = X, Y = Y, Z = Z, r = r, mu = mu, X_test = X_test, Y_test =
              Y_test, Z_test = Z_test, r_test = r_test, mu_test = mu_test))
}

## Simiulate dataset
sim_data <- gen_data(250, 250, 100, 1)

df <- data.frame(X = sim_data$X, Y = sim_data$Y, Z = sim_data$Z)
df_test <- data.frame(X = sim_data$X_test, Y = sim_data$Y_test, Z = sim_data$Z_test)

## Fit the model

opts <- Opts(num_burn = num_burn, num_save = num_save)
fitted_vc <- vc_softbart_regression(Y ~ . -Z, "Z", df, df_test, opts = opts)

## Plot results

plot(colMeans(fitted_vc$mu_test), sim_data$mu_test)
abline(a = 0, b = 1)

}
