% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/coef.R
\name{coef.svem_model}
\alias{coef.svem_model}
\title{Coefficients for SVEM Models}
\usage{
\method{coef}{svem_model}(object, debiased = FALSE, ...)
}
\arguments{
\item{object}{An object of class \code{svem_model}, typically returned by
\code{\link{SVEMnet}}.}

\item{debiased}{Logical; if \code{TRUE} and debiased coefficients are
available (Gaussian fits with \code{parms_debiased}), return those instead
of \code{parms}. Default is \code{FALSE}.}

\item{...}{Unused; present for S3 method compatibility.}
}
\value{
A named numeric vector of coefficients (including the intercept).
}
\description{
Extracts averaged coefficients from an \code{svem_model} fitted by
\code{\link{SVEMnet}}.
}
\details{
For Gaussian fits, you can optionally request debiased coefficients (if they
were computed and stored) via \code{debiased = TRUE}. In that case, the
function returns \code{object$parms_debiased}. If debiased coefficients are
not available, or if \code{debiased = FALSE}, the function returns
\code{object$parms}, which are the ensemble-averaged coefficients across
bootstrap members.

For Binomial models, \code{debiased} is ignored and the averaged
coefficients in \code{object$parms} are returned.

This is a lightweight accessor around the stored components of an
\code{svem_model}:
\itemize{
\item \code{parms}: ensemble-averaged coefficients over bootstrap members,
on the model's link scale;
\item \code{parms_debiased}: optional debiased coefficients (Gaussian only),
if requested at fit time.
}
Passing \code{debiased = TRUE} has no effect if \code{parms_debiased} is
\code{NULL}.
}
\examples{
\donttest{
  set.seed(1)
  n  <- 200
  x1 <- rnorm(n)
  x2 <- rnorm(n)
  eps <- rnorm(n, sd = 0.3)
  y_g <- 1 + 2*x1 - 0.5*x2 + eps
  dat_g <- data.frame(y_g, x1, x2)

  # Small nBoot to keep runtime light in examples
  fit_g <- SVEMnet(y_g ~ x1 + x2, data = dat_g, nBoot = 30, relaxed = TRUE)

  # Ensemble-averaged coefficients
  cc <- coef(fit_g)
  head(cc)

  # Debiased (only if available for Gaussian fits)
  ccd <- coef(fit_g, debiased = TRUE)
  head(ccd)

  # Binomial example (0/1 outcome)
  set.seed(2)
  n  <- 250
  x1 <- rnorm(n)
  x2 <- rnorm(n)
  eta <- -0.4 + 1.1*x1 - 0.7*x2
  p   <- 1/(1+exp(-eta))
  y_b <- rbinom(n, 1, p)
  dat_b <- data.frame(y_b, x1, x2)

  fit_b <- SVEMnet(y_b ~ x1 + x2, data = dat_b,
                   family = "binomial", nBoot = 30, relaxed = TRUE)

  # Averaged coefficients (binomial; debiased is ignored)
  coef(fit_b)
}

}
\seealso{
\code{\link{svem_nonzero}} for bootstrap nonzero percentages and a
quick stability plot.
}
