% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/peer.ces.R
\name{cespeer}
\alias{cespeer}
\title{Estimation of CES-Based Peer Effects Models}
\usage{
cespeer(
  formula,
  instrument,
  Glist,
  structural = FALSE,
  fixed.effects = FALSE,
  set.rho = NULL,
  grid.rho = seq(-400, 400, radius),
  radius = 5,
  tol = 1e-08,
  drop = NULL,
  compute.cov = TRUE,
  HAC = "iid",
  data
)
}
\arguments{
\item{formula}{An object of class \link[stats]{formula}: a symbolic description of the model. \code{formula} should be specified as \code{y ~ x1 + x2},
where \code{y} is the outcome and \code{x1} and \code{x2} are control variables, which can include contextual variables such as averages or quantiles among peers.}

\item{instrument}{An object of class \link[stats]{formula} indicating the excluded instrument. It should be specified as \code{~ z},
where \code{z} is the excluded instrument for the outcome. Following Boucher et al. (2024), it can be an OLS exogenous prediction of \code{y}.
This prediction is used to compute instruments for the CES function of peer outcomes.}

\item{Glist}{The adjacency matrix. For networks consisting of multiple subnets (e.g., schools), \code{Glist} must be a list of subnets, with the \code{m}-th element being an \eqn{n_m \times n_m} adjacency matrix, where \eqn{n_m} is the number of nodes in the \code{m}-th subnet.}

\item{structural}{A logical value indicating whether the reduced-form or structural specification should be estimated (see details).}

\item{fixed.effects}{A logical value or string specifying whether the model includes subnet fixed effects. The fixed effects may differ between isolated and non-isolated nodes. Accepted values are \code{"no"} or \code{"FALSE"} (indicating no fixed effects),
\code{"join"} or \code{TRUE} (indicating the same fixed effects for isolated and non-isolated nodes within each subnet), and \code{"separate"} (indicating different fixed effects for isolated and non-isolated nodes within each subnet). Note that \code{"join"} fixed effects are not applicable for structural models;
\code{"join"} and \code{TRUE} are automatically converted to \code{"separate"}.}

\item{set.rho}{A fixed value for the CES substitution parameter to estimate a constrained model. Given this value, the other parameters can be estimated.}

\item{grid.rho}{A finite grid of values for the CES substitution parameter \eqn{\rho} (see Details).
This grid is used to obtain the starting value and define the GMM weight.
It is recommended to use a finely subdivided grid.}

\item{radius}{The radius of the subset in which the estimate for \eqn{\rho} is determined.
The subset is a segment centered at the optimal \eqn{\rho} found using \code{grid.rho}.
For better numerical optimization performance, use a finely subdivided \code{grid.rho} and a small \code{radius}.}

\item{tol}{A tolerance value used in the QR factorization to identify columns of explanatory variable and instrument matrices that ensure a full-rank matrix (see the \link[base]{qr} function).
The same tolerance is also used in the to minimize the concentrated GMM objective function (see \link[stats]{optimise}).}

\item{drop}{A dummy vector of the same length as the sample, indicating whether an observation should be dropped.
This can be used, for example, to remove false isolates or to estimate the model only on non-isolated agents.
These observations cannot be directly removed from the network by the user because they may still be friends with other agents.}

\item{compute.cov}{A logical value indicating whether the covariance matrix of the estimator should be computed.}

\item{HAC}{A character string specifying the correlation structure among the idiosyncratic error terms for covariance computation. Options are \code{"iid"} for independent errors, \code{"hetero"} for heteroskedastic non-autocorrelated errors, and \code{"cluster"} for heteroskedastic errors with potential within-subnet correlation.}

\item{data}{An optional data frame, list, or environment (or an object that can be coerced by \link[base]{as.data.frame} to a data frame) containing the variables
in the model. If not found in \code{data}, the variables are taken from \code{environment(formula)}, typically the environment from which \code{cespeer} is called.}
}
\value{
A list containing:
\item{model.info}{A list with information about the model, including the number of subnets, the number of observations, and other key details.}
\item{gmm}{A list of GMM estimation results, including parameter estimates, the covariance matrix, and related statistics.}
\item{first.search}{A list containing initial estimations on the grid of values for \eqn{\rho}.}
}
\description{
\code{cespeer} estimates the CES-based peer effects model introduced by Boucher et al. (2024). See Details.
}
\details{
Let \eqn{\mathcal{N}} be a set of \eqn{n} agents indexed by the integer \eqn{i \in [1, n]}.
Agents are connected through a network characterized by an adjacency matrix \eqn{\mathbf{G} = [g_{ij}]} of dimension \eqn{n \times n}, where \eqn{g_{ij} = 1} if agent \eqn{j} is a friend of agent \eqn{i}, and \eqn{g_{ij} = 0} otherwise.
In weighted networks, \eqn{g_{ij}} can be a nonnegative variable (not necessarily binary) that measures the intensity of the outgoing link from \eqn{i} to \eqn{j}. The model can also accommodate such networks.
Note that the network generally consists of multiple independent subnets (e.g., schools).
The \code{Glist} argument is the list of subnets. In the case of a single subnet, \code{Glist} should be a list containing one matrix.\cr

The reduced-form specification of the CES-based peer effects model is given by:
\deqn{y_i = \lambda\left(\sum_{j = 1}^n g_{ij}y_j^{\rho}\right)^{1/\rho} + \mathbf{x}_i^{\prime}\beta + \varepsilon_i,}
where \eqn{\varepsilon_i} is an idiosyncratic error term, \eqn{\lambda} captures the effect of the social norm \eqn{\left(\sum_{j = 1}^n g_{ij}y_j^{\rho}\right)^{1/\rho}},
and \eqn{\beta} captures the effect of \eqn{\mathbf{x}_i} on \eqn{y_i}. The parameter \eqn{\rho} determines the form of the social norm in the model.
\itemize{
\item When \eqn{\rho > 1}, individuals are more sensitive to peers with high outcomes.
\item When \eqn{\rho < 1}, individuals are more sensitive to peers with low outcomes.
\item When \eqn{\rho = 1}, peer effects are uniform across peer outcome values.\cr
}

The structural specification of the model differs for isolated and non-isolated individuals.
For an \strong{isolated} \eqn{i}, the specification is similar to a standard linear-in-means model without social interactions, given by:
\deqn{y_i = \mathbf{x}_i^{\prime}\beta + \varepsilon_i.}
If node \eqn{i} is \strong{non-isolated}, the specification is:
\deqn{y_i = \lambda\left(\sum_{j = 1}^n g_{ij}y_j^{\rho}\right)^{1/\rho} + (1 - \lambda_2)\mathbf{x}_i^{\prime}\beta + \varepsilon_i,}
where \eqn{\lambda_2} determines whether preferences exhibit conformity or complementarity/substitution.
Identification of \eqn{\beta} and \eqn{\lambda_2} requires the network to include a sufficient number of isolated individuals.
}
\examples{
\donttest{
set.seed(123)
ngr  <- 50  # Number of subnets
nvec <- rep(30, ngr)  # Size of subnets
n    <- sum(nvec)

### Simulating Data
## Network matrix
G <- lapply(1:ngr, function(z) {
  Gz <- matrix(rbinom(nvec[z]^2, 1, 0.3), nvec[z], nvec[z])
  diag(Gz) <- 0
  # Adding isolated nodes (important for the structural model)
  niso <- sample(0:nvec[z], 1, prob = (nvec[z] + 1):1 / sum((nvec[z] + 1):1))
  if (niso > 0) {
    Gz[sample(1:nvec[z], niso), ] <- 0
  }
  # Row-normalization
  rs   <- rowSums(Gz); rs[rs == 0] <- 1
  Gz/rs
})

X   <- cbind(rnorm(n), rpois(n, 2))
l   <- 0.55
b   <- c(2, -0.5, 1)
rho <- -2
eps <- rnorm(n, 0, 0.4)

## Generating `y`
y <- cespeer.sim(formula = ~ X, Glist = G, rho = rho, lambda = l,
                 beta = b, epsilon = eps)$y

### Estimation
## Computing instruments
z <- fitted.values(lm(y ~ X))

## Reduced-form model
rest <- cespeer(formula = y ~ X, instrument = ~ z, Glist = G, fixed.effects = "yes",
                radius = 5, grid.rho = seq(-10, 10, 1))
summary(rest)

## Structural model
sest <- cespeer(formula = y ~ X, instrument = ~ z, Glist = G, fixed.effects = "yes",
                radius = 5, structural = TRUE, grid.rho = seq(-10, 10, 1))
summary(sest)

## Quantile model
z    <- qpeer.inst(formula = ~ X, Glist = G, tau = seq(0, 1, 0.1), max.distance = 2, 
                   checkrank = TRUE)$instruments
qest <- qpeer(formula = y ~ X, excluded.instruments  = ~ z, Glist = G, 
              fixed.effects = "yes", tau = seq(0, 1, 1/3), structural = TRUE)
summary(qest)
}
}
\references{
Boucher, V., Rendall, M., Ushchev, P., & Zenou, Y. (2024). Toward a general theory of peer effects. Econometrica, 92(2), 543-565, \doi{10.3982/ECTA21048}.
}
\seealso{
\code{\link{qpeer}}, \code{\link{linpeer}}
}
