#' Calculate Master Mix Reagents
#'
#' Generates a recipe for your qPCR Master Mix based on sample count.
#'
#' @param n_samples Number of biological samples.
#' @param n_genes Number of genes (targets + reference).
#' @param replicates Number of technical replicates (usually 3).
#' @param rxn_volume Total volume per well (e.g., 20 uL).
#' @param dead_volume_pct Percentage of extra mix to prepare for pipetting error (default 10%).
#' @return A data frame containing the mix recipe.
#' @export
#' @examples
#' # Plan for 12 samples, 2 genes, standard 20uL reaction
#' calculate_reagents(n_samples = 12, n_genes = 2)
#'
#' # Adjust for a 10uL reaction volume and 15% dead volume
#' calculate_reagents(12, 2, rxn_volume = 10, dead_volume_pct = 15)
calculate_reagents <- function(n_samples, n_genes, replicates = 3, rxn_volume = 20, dead_volume_pct = 10) {

  # Calculate total wells needed per gene (+2 for NTC)
  total_wells <- (n_samples * replicates) + 2

  # Add Dead Volume Buffer
  buffer_multiplier <- 1 + (dead_volume_pct / 100)
  wells_to_make <- ceiling(total_wells * buffer_multiplier)

  # Reagent Volumes
  sybr_vol <- (rxn_volume / 2) * wells_to_make
  primer_vol <- (rxn_volume * 0.05) * wells_to_make
  water_vol  <- (rxn_volume * 0.35) * wells_to_make

  # Create Recipe Table
  recipe <- data.frame(
    Reagent = c("2x Master Mix (SYBR)", "Fwd Primer (10uM)", "Rev Primer (10uM)", "Nuclease-Free Water", "--- TOTAL PER GENE ---"),
    Volume_uL = c(sybr_vol, primer_vol, primer_vol, water_vol, sum(sybr_vol, primer_vol*2, water_vol))
  )

  message(paste("Plan for:", n_samples, "samples,", n_genes, "genes."))
  message(paste("Preparing for", wells_to_make, "reactions per gene (includes buffer)."))

  return(recipe)
}
