% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/obc_sblp.R
\name{ob_categorical_sblp}
\alias{ob_categorical_sblp}
\title{Optimal Binning for Categorical Variables using SBLP}
\usage{
ob_categorical_sblp(
  feature,
  target,
  min_bins = 3L,
  max_bins = 5L,
  bin_cutoff = 0.05,
  max_n_prebins = 20L,
  convergence_threshold = 1e-06,
  max_iterations = 1000L,
  bin_separator = "\%;\%",
  alpha = 0.5
)
}
\arguments{
\item{feature}{A character vector or factor representing the categorical
predictor variable. Missing values (NA) will be converted to the string
"NA" and treated as a separate category.}

\item{target}{An integer vector containing binary outcome values (0 or 1).
Must be the same length as \code{feature}. Cannot contain missing values.}

\item{min_bins}{Integer. Minimum number of bins to create. Must be at least
2. Default is 3.}

\item{max_bins}{Integer. Maximum number of bins to create. Must be greater
than or equal to \code{min_bins}. Default is 5.}

\item{bin_cutoff}{Numeric. Minimum relative frequency threshold for
individual categories. Categories with frequency below this proportion
will be merged with similar categories before the main optimization.
Value must be between 0 and 1. Default is 0.05 (5\%).}

\item{max_n_prebins}{Integer. Maximum number of initial bins/groups allowed
before the dynamic programming optimization. If the number of unique
categories exceeds this, similar adjacent categories are pre-merged.
Default is 20.}

\item{convergence_threshold}{Numeric. Threshold for determining algorithm
convergence based on changes in total Information Value. Default is 1e-6.}

\item{max_iterations}{Integer. Maximum number of iterations for the
optimization process. Default is 1000.}

\item{bin_separator}{Character string used to separate category names when
multiple categories are merged into a single bin. Default is "\%;\%".}

\item{alpha}{Numeric. Laplace smoothing parameter added to counts to avoid
division by zero and stabilize WoE calculations for sparse data. Must be
non-negative. Default is 0.5.}
}
\value{
A list containing the results of the optimal binning procedure:
\describe{
  \item{\code{id}}{Numeric vector of bin identifiers (1 to n_bins)}
  \item{\code{bin}}{Character vector of bin labels, which are combinations
        of original categories separated by \code{bin_separator}}
  \item{\code{woe}}{Numeric vector of Weight of Evidence values for each bin}
  \item{\code{iv}}{Numeric vector of Information Values for each bin}
  \item{\code{count}}{Integer vector of total observations in each bin}
  \item{\code{count_pos}}{Integer vector of positive outcomes in each bin}
  \item{\code{count_neg}}{Integer vector of negative outcomes in each bin}
  \item{\code{rate}}{Numeric vector of the observed event rate in each bin}
  \item{\code{total_iv}}{Numeric scalar. Total Information Value across all
        bins}
  \item{\code{converged}}{Logical. Whether the algorithm converged}
  \item{\code{iterations}}{Integer. Number of iterations performed}
}
}
\description{
This function performs optimal binning for categorical variables using the
Similarity-Based Logistic Partitioning (SBLP) algorithm. This approach
combines logistic properties (sorting categories by event rate) with dynamic
programming to find the optimal partition that maximizes Information Value (IV).
}
\details{
The SBLP algorithm follows these steps:
\enumerate{
  \item \strong{Preprocessing}: Handling of missing values and calculation of
        initial statistics.
  \item \strong{Rare Category Consolidation}: Categories with frequency below
        \code{bin_cutoff} are merged with statistically similar categories based
        on their target rates.
  \item \strong{Sorting}: Unique categories (or merged groups) are sorted by
        their empirical event rate (probability of target=1).
  \item \strong{Dynamic Programming}: An optimal partitioning algorithm
        (similar to Jenks Natural Breaks but optimizing IV) is applied to the
        sorted sequence to determine the cutpoints that maximize the total IV.
  \item \strong{Refinement}: Post-processing ensures constraints like monotonicity
        and minimum bin size are met.
}

A key feature of this implementation is the use of \strong{Laplace Smoothing}
(controlled by the \code{alpha} parameter) to prevent infinite WoE values
and stabilize estimates for categories with small counts.

Mathematical definitions with smoothing:

The smoothed event rate \eqn{p_i} for a bin is calculated as:
\deqn{p_i = \frac{n_{pos} + \alpha}{n_{total} + 2\alpha}}{
p_i = (n_pos + alpha) / (n_total + 2*alpha)}

The Weight of Evidence (WoE) is computed using smoothed proportions:
\deqn{WoE_i = \ln\left(\frac{p_i^{(1)}}{p_i^{(0)}}\right)}{
WoE_i = ln(prob_pos / prob_neg)}

where \eqn{p_i^{(1)}}{prob_pos} and \eqn{p_i^{(0)}}{prob_neg} are the
smoothed distributions of positive and negative classes across bins.
}
\note{
\itemize{
  \item Target variable must contain both 0 and 1 values.
  \item Unlike heuristic methods, this algorithm uses Dynamic Programming
        which guarantees an optimal partition given the sorted order of
        categories.
  \item Monotonicity is generally enforced by the sorting step, but strictly
        checked and corrected in the final output.
}
}
\examples{
# Generate sample data
set.seed(123)
n <- 1000
feature <- sample(letters[1:8], n, replace = TRUE)
# Create a relationship where 'a' and 'b' have high probability
target <- rbinom(n, 1, prob = ifelse(feature \%in\% c("a", "b"), 0.8, 0.2))

# Perform optimal binning
result <- ob_categorical_sblp(feature, target)
print(result[c("bin", "woe", "iv", "count")])

# Using a higher smoothing parameter (alpha)
result_smooth <- ob_categorical_sblp(
  feature = feature,
  target = target,
  alpha = 1.0
)

# Handling missing values
feature_with_na <- feature
feature_with_na[sample(length(feature_with_na), 50)] <- NA
result_na <- ob_categorical_sblp(feature_with_na, target)

}
