# JapanAPIs - Access Japanese Data via Public APIs and Curated Datasets
# Version 0.1.1
# Copyright (c) 2025 Renzo Caceres Rossi
# Licensed under the MIT License.
# See the LICENSE file in the root directory for full license text.

# get_japan_hospital_beds

library(testthat)

test_that("get_japan_hospital_beds() returns a tibble with expected structure", {
  skip_on_cran()

  result <- get_japan_hospital_beds()

  # Should return tibble or NULL if API down
  expect_true(is.null(result) || inherits(result, "tbl_df"))

  # If NULL, stop further checks
  if (is.null(result)) return(invisible(TRUE))

  # Structure
  expect_s3_class(result, "tbl_df")
  expect_equal(ncol(result), 4)

  # Column names
  expect_named(result, c("indicator", "country", "year", "value"))

  # Column types
  expect_type(result$indicator, "character")
  expect_type(result$country, "character")
  expect_type(result$year, "integer")
  expect_type(result$value, "double")
})

test_that("get_japan_hospital_beds() returns correct dimensions and years", {
  skip_on_cran()

  result <- get_japan_hospital_beds()
  if (is.null(result)) return(invisible(TRUE))

  # 2010–2022 inclusive → 13 rows
  expect_equal(nrow(result), 13)

  # Expected years
  expect_equal(sort(result$year), 2010:2022)

  # Should be sorted in descending order
  expect_true(all(diff(result$year) <= 0))
})

test_that("get_japan_hospital_beds() returns consistent values for Japan", {
  skip_on_cran()

  result <- get_japan_hospital_beds()
  if (is.null(result)) return(invisible(TRUE))

  # Country should always be Japan
  expect_true(all(result$country == "Japan"))

  # Indicator should always match hospital beds
  expect_true(all(grepl("Hospital beds", result$indicator, ignore.case = TRUE)))

  # Values should be non-negative
  expect_true(all(result$value >= 0, na.rm = TRUE))

  # Japan historically has very high hospital bed numbers
  # Reasonable bounds: between 10 and 20 beds per 1,000 people
  expect_true(all(result$value[result$year <= 2019] >= 10, na.rm = TRUE))
  expect_true(all(result$value[result$year <= 2019] <= 20, na.rm = TRUE))
})

test_that("get_japan_hospital_beds() handles missing values correctly", {
  skip_on_cran()

  result <- get_japan_hospital_beds()
  if (is.null(result)) return(invisible(TRUE))

  # Last two years (2021–2022) are currently NA → expected
  expect_true(any(is.na(result$value)))

  # Years with known values must not be NA
  known_years <- result$year[result$year <= 2020]
  known_values <- result$value[result$year <= 2020]
  expect_true(all(!is.na(known_values)))
})

test_that("get_japan_hospital_beds() handles API errors gracefully", {
  skip_on_cran()

  result <- tryCatch(get_japan_hospital_beds(), error = function(e) NULL)

  # Must return NULL or tibble
  expect_true(is.null(result) || inherits(result, "tbl_df"))
})
