/*
 * Copyright (C) 2023, KylinSoft Co., Ltd.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses/>.
 *
**/
#include "areaui.h"

AreaUi::AreaUi(QWidget *parent, Qt::WindowFlags f)
    : QWidget(parent, f)
{
    setAttribute(Qt::WA_DeleteOnClose);
    uiLayout       = new QVBoxLayout(this);
    formatLayout   = new QVBoxLayout(this);
    LanguageLayout = new QVBoxLayout(this);
    uiLayout->addLayout(formatLayout);
    uiLayout->addLayout(LanguageLayout);
    uiLayout->addStretch();
    uiLayout->setSpacing(40);
    uiLayout->setMargin(0);
    formatLayout->setSpacing(8);
    LanguageLayout->setSpacing(1);
    formatLayout->setMargin(0);
    LanguageLayout->setMargin(0);
    initUi();
}

AreaUi::~AreaUi()
{

}

void AreaUi::initUi()
{
    initLanguageFormat();
    initSystemLanguage();
}

void AreaUi::initLanguageFormat()
{
    formatTitleLabel = new TitleLabel(this);
    formatGroup      = new SettingGroup(this);
    //~ contents_path /Area/Regional Format
    countryWidget    = new ComboxWidget(tr("Regional Format"), formatGroup, UkccFrame::BorderRadiusStyle::Top);
    //~ contents_path /Area/Calendar
    calendarWidget   = new ComboxWidget(tr("Calendar"), formatGroup, UkccFrame::BorderRadiusStyle::None);
    //~ contents_path /Area/First Day Of Week
    dayWidget        = new ComboxWidget(tr("First Day Of Week"), formatGroup, UkccFrame::BorderRadiusStyle::None);
    //~ contents_path /Area/Date
#ifdef Nile
    shortDateWidget  = new ComboxWidget(tr("Short Foramt Date"), formatGroup, UkccFrame::BorderRadiusStyle::None);
    longDateWidget   = new ComboxWidget(tr("Long Format Date"), formatGroup, UkccFrame::BorderRadiusStyle::None);
#else
    shortDateWidget  = new ComboxWidget(tr("Date"), formatGroup, UkccFrame::BorderRadiusStyle::None);
#endif

    //~ contents_path /Area/Time
    timeWidget       = new ComboxWidget(tr("Time"), formatGroup, UkccFrame::BorderRadiusStyle::Bottom);

    formatLayout->addWidget(formatTitleLabel);
    formatLayout->addWidget(formatGroup);
    formatGroup->insertWidget(COUNTRY, countryWidget);
    formatGroup->insertWidget(CALENDAR, calendarWidget);
    formatGroup->insertWidget(FIRSTDAY, dayWidget);
    formatGroup->insertWidget(DATEFORMAT, shortDateWidget);
#ifdef Nile
    formatGroup->insertWidget(DATEFORMAT, longDateWidget);
#endif
    formatGroup->insertWidget(TIMEFORMAT, timeWidget);
    //~ contents_path /Area/Language Format
    formatTitleLabel->setText(tr("Language Format"));
}

void AreaUi::initSystemLanguage()
{
    languageTitleLable   = new TitleLabel(this);
    languageGroup        = new SettingGroup(this);
    addLanguageBtn       = new AddButton(this, UkccFrame::BorderRadiusStyle::Bottom);
    addLanguageDialog    = new AddLanguageDialog(this);
    addInputMehtodDialog = new AddInputMethodDialog(this);

    LanguageLayout->addWidget(languageTitleLable);
    LanguageLayout->addSpacerItem(new QSpacerItem(8, 7, QSizePolicy::Fixed, QSizePolicy::Fixed));
    LanguageLayout->addWidget(languageGroup);
    LanguageLayout->addWidget(addLanguageBtn);
    languageGroup->setOneRadiusStyle(UkccFrame::BorderRadiusStyle::Top);
    languageGroup->setLastRadiusStyle(UkccFrame::BorderRadiusStyle::None);
    //~ contents_path /Area/System Language
    languageTitleLable->setText(tr("System Language"));
    addLanguageDialog->setHidden(true);
    addInputMehtodDialog->setHidden(true);
}

void AreaUi::addShowLanguage(QString languageCode, QString languageName, bool selected)
{
    bool isAdded = false;
    for (LanguageFrame *f : languageFrameList) {
        if (f->getShowName() == languageName) {
            isAdded = true;
        }
    }
    if (isAdded) {
        return;
    } else {
        LanguageFrame *languageF = new LanguageFrame(languageName, languageGroup);
        languageF->showSelectedIcon(selected);
        languageGroup->addWidget(languageF);
        languageFrameList.append(languageF);
        connect(languageF, &LanguageFrame::clicked, this, [=](){
            Q_EMIT languageChanged(languageCode);
            for (LanguageFrame *f : languageFrameList) {
                if (f->getShowName() != languageName) {
                    f->showSelectedIcon(false);
                }
            }
            languageF->showSelectedIcon(true);
        });
        connect(languageF, &LanguageFrame::deleted, this, [=]() {
            Q_EMIT showLanguageListRemoved(languageCode);
            languageGroup->removeWidget(languageF);
            languageFrameList.removeOne(languageF);
        });
    }
}

void AreaUi::clearShowLanguage()
{
    qDeleteAll(languageFrameList);
    languageFrameList.clear();
}

void AreaUi::reloadLanguageFrameIcon()
{
    for (LanguageFrame *f : languageFrameList) {
        f->reloadIcon();
    }
}

void AreaUi::settingForIntel()
{
    countryWidget->hide();
    dayWidget->hide();
    languageTitleLable->hide();
    addLanguageBtn->hide();
    languageGroup->hide();
}
