
#' Extract Residuals from an snreg Model
#'
#' @title Residuals for snreg Objects
#'
#' @description
#' \code{residuals.snreg} is the S3 method for extracting residuals from a fitted
#' \code{snreg} model. Residuals may be returned either for the full data or only
#' for the estimation sample.
#'
#' @param object
#' an object of class \code{"snreg"}, typically produced by \code{\link{snreg}}.
#'
#' @param esample
#' logical. If \code{TRUE} (default), residuals are returned only for observations
#' used in estimation (others are \code{NA}).  
#' If \code{FALSE}, the raw vector of residuals (\code{obj$resid}) is returned.
#'
#' @param ...
#' additional arguments (currently unused).
#'
#' @return
#' A numeric vector of residuals.  
#' If \code{esample = TRUE}, the vector matches the length of the original data
#' and contains \code{NA} for non-estimation observations.  
#' If \code{esample = FALSE}, only the computed residuals are returned.
#'
#' @details
#' This method simply accesses the \code{obj$resid} component of a fitted
#' \code{"snreg"} object. An informative error is produced if residuals are not
#' available.
#'
#' @seealso
#' \code{\link{snsf}}, \code{\link{snreg}}, \code{\link{lm.mle}}, \code{\link{vcov.snreg}}, \code{\link{coef.snreg}}
#'
#' @examples
#' library(snreg)
#' 
#' data("banks07")
#' head(banks07)
#' 
#' # Translog cost function specification
#' 
#' spe.tl <- log(TC) ~ (log(Y1) + log(Y2) + log(W1) + log(W2))^2 +
#'   I(0.5 * log(Y1)^2) + I(0.5 * log(Y2)^2) +
#'   I(0.5 * log(W1)^2) + I(0.5 * log(W2)^2)
#' 
#' # Specification 1: homoskedastic noise and skewness
#' 
#' formSV <- NULL   # variance equation; constant variance
#' formSK <- NULL   # skewness equation; constant skewness
#' 
#' m1 <- snreg(
#'   formula  = spe.tl,
#'   data     = banks07,
#'   ln.var.v = formSV,
#'   skew.v   = formSK
#' )
#' 
#' residuals(m1)
#'
#' @export
residuals.snreg <- function(object, esample = TRUE, ...) {
  
  if (is.null(object)) {
    stop("Argument 'object' is NULL; expected a fitted 'snreg' objectect.", call. = FALSE)
  }
  if (is.null(object$resid)) {
    stop("No residuals are available in 'object$resid'.", call. = FALSE)
  }
  
  if (isTRUE(esample)) {
    if (is.null(object$esample)) {
      stop("objectect does not contain an 'esample' indicator.", call. = FALSE)
    }
    out <- rep(NA_real_, length(object$esample))
    out[object$esample] <- object$resid
    return(out)
  }
  
  # esample = FALSE → return raw residuals
  object$resid
}


# residuals.snreg <- function( obj, esample = TRUE, ... ) {
#   if(is.null(obj$resid) | is.null(obj)){
#     stop( paste("No residuals are available in ",obj,"") )
#   }
#   if(esample){
#     my.res <- rep(NA, length(obj$esample))
#     my.res[obj$esample] <- obj$resid
#   } else {
#     my.res <- obj$resid
#   }
#   return( my.res )
# }