#' @keywords internal
resolve_tree_path <- function(path,
                              project = c("auto", "root", "none"),
                              search_paths = c(".", ".."),
                              root_markers = c(".Rproj", "DESCRIPTION")) {
  project <- match.arg(project)

  path <- path.expand(path)
  search_paths <- path.expand(search_paths)

  # If a .Rproj file is supplied, treat its directory as the project path
  if (file.exists(path) && grepl("\\.Rproj$", basename(path), ignore.case = TRUE)) {
    path <- dirname(normalizePath(path, winslash = "/", mustWork = TRUE))
  }

  # If not a directory, try to resolve by searching common locations
  if (!dir.exists(path)) {
    path <- find_project_by_name(path, search_paths = search_paths)
  }

  path <- normalizePath(path, winslash = "/", mustWork = TRUE)

  if (project == "none") return(path)

  if (project == "auto") {
    # Try to find a root above; if none found, keep the original path
    root <- find_project_root_up(path, root_markers = root_markers)
    if (!identical(normalizePath(root, winslash = "/", mustWork = TRUE), path)) {
      return(root)
    }
    return(path)
  }

  # project == "root"
  find_project_root_up(path, root_markers = root_markers)
}

#' @keywords internal
find_project_by_name <- function(name, search_paths = c(".", "..")) {
  name <- path.expand(name)
  search_paths <- path.expand(search_paths)
  search_paths <- search_paths[dir.exists(search_paths)]

  if (dir.exists(name)) return(name)

  if (file.exists(name) && grepl("\\.Rproj$", basename(name), ignore.case = TRUE)) {
    return(dirname(normalizePath(name, winslash = "/", mustWork = TRUE)))
  }

  is_rproj_name <- grepl("\\.Rproj$", name, ignore.case = TRUE)
  rproj_file <- if (is_rproj_name) name else paste0(name, ".Rproj")

  for (base in search_paths) {
    dir_candidate <- file.path(base, name)
    if (dir.exists(dir_candidate)) return(dir_candidate)

    rproj_candidate <- file.path(base, rproj_file)
    if (file.exists(rproj_candidate)) {
      return(dirname(normalizePath(rproj_candidate, winslash = "/", mustWork = TRUE)))
    }

    rproj_inside_dir <- file.path(base, name, basename(rproj_file))
    if (file.exists(rproj_inside_dir)) {
      return(dirname(normalizePath(rproj_inside_dir, winslash = "/", mustWork = TRUE)))
    }
  }

  stop(
    "Could not find directory or .Rproj for '", name, "' in search_paths.\n",
    "Try an explicit path or set search_paths.",
    call. = FALSE
  )
}

#' @keywords internal
has_root_marker <- function(dir, root_markers) {
  if (!length(root_markers)) return(FALSE)
  if (!dir.exists(dir)) return(FALSE)

  for (m in root_markers) {
    if (identical(m, ".Rproj")) {
      files <- safe_list(dir, show_hidden = TRUE)
      if (any(grepl("\\.Rproj$", basename(files), ignore.case = TRUE))) return(TRUE)
    } else {
      if (file.exists(file.path(dir, m))) return(TRUE)
    }
  }

  FALSE
}

#' @keywords internal
find_project_root_up <- function(path, root_markers = c(".Rproj", "DESCRIPTION")) {
  current <- normalizePath(path, winslash = "/", mustWork = TRUE)

  repeat {
    if (has_root_marker(current, root_markers)) return(current)

    parent <- dirname(current)
    if (identical(parent, current)) break
    current <- parent
  }

  path
}

#' @keywords internal
safe_list <- function(path, show_hidden = FALSE) {
  tryCatch(
    list.files(path, full.names = TRUE, all.files = show_hidden, no.. = TRUE),
    error = function(e) character(0)
  )
}
