#' Train a similarity-based scouting model
#'
#' This function prepares numeric player features for similarity-based
#' scouting analysis.
#'
#' @param df A cleaned data frame containing player information.
#'
#' @return A list containing:
#' \describe{
#'   \item{data}{A numeric matrix of standardized player features.}
#'   \item{players}{Character vector of player names.}
#' }
#'
#' @details
#' The returned object is intended to be used as input for
#' \code{get_similar_players()}.
#'
#' @examples
#' df <- data.frame(
#'   name = c("Player A", "Player B"),
#'   age = c(21, 23),
#'   market_value_est = c(500, 750),
#'   club = c("Club A", "Club B"),
#'   league_country = c("Indonesia", "Indonesia"),
#'   stringsAsFactors = FALSE
#' )
#'
#' model <- train_scout_brain(df)
#'
#' @export
train_scout_brain <- function(df) {

  features <- df |> dplyr::select(age, market_value_est)
  scaled <- scale(features)

  list(
    data = df,
    features = scaled
  )
}

#' Retrieve similar players based on cosine similarity
#'
#' @param model A trained scouting model returned by
#'   \code{train_scout_brain()}.
#' @param player_name Character string specifying the reference player.
#' @param top_n Integer indicating the number of similar players to return.
#'
#' @return A data frame with similarity scores for the most similar players.
#'
#' @details
#' Similarity is computed using cosine similarity on standardized
#' numeric features. The reference player is excluded from the results.
#'
#' @examples
#' df <- data.frame(
#'   name = c("Player A", "Player B", "Player C"),
#'   age = c(21, 23, 22),
#'   market_value_est = c(500, 750, 600),
#'   club = c("Club A", "Club B", "Club C"),
#'   league_country = c("Indonesia", "Indonesia", "Indonesia"),
#'   stringsAsFactors = FALSE
#' )
#'
#' model <- train_scout_brain(df)
#' get_similar_players(model, "Player A", top_n = 2)
#'
#' @export
get_similar_players <- function(model, player_name, top_n = 5) {

  idx <- which(model$data$name == player_name)
  if (length(idx) == 0) stop("Player not found")

  sim <- proxy::simil(
    model$features,
    model$features[idx, , drop = FALSE],
    method = "cosine"
  )

  model$data |>
    dplyr::mutate(similarity = as.numeric(sim)) |>
    dplyr::arrange(dplyr::desc(similarity)) |>
    dplyr::slice(2:(top_n + 1))
}
