% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/utils-target-data.R
\name{target-data-utils}
\alias{target-data-utils}
\alias{get_date_col}
\alias{get_observable_unit}
\alias{get_versioned}
\alias{get_has_output_type_ids}
\alias{get_non_task_id_schema}
\alias{has_target_data_config}
\alias{has_target_data_config.default}
\alias{has_target_data_config.SubTreeFileSystem}
\title{Get target data configuration properties}
\usage{
get_date_col(config_target_data)

get_observable_unit(
  config_target_data,
  dataset = c("time-series", "oracle-output")
)

get_versioned(config_target_data, dataset = c("time-series", "oracle-output"))

get_has_output_type_ids(config_target_data)

get_non_task_id_schema(config_target_data)

has_target_data_config(hub_path)

\method{has_target_data_config}{default}(hub_path)

\method{has_target_data_config}{SubTreeFileSystem}(hub_path)
}
\arguments{
\item{config_target_data}{A target-data config object created by
\code{read_config(hub_path, "target-data")}.}

\item{dataset}{Character string specifying the dataset type: either
\code{"time-series"} or \code{"oracle-output"}. Used for functions that extract
dataset-specific properties.}

\item{hub_path}{Path to a hub. Can be a local directory path or cloud URL
(S3, GCS).}
}
\value{
\code{get_date_col()} returns a character string: the name of the date
column that stores the date on which observed data actually occurred.

\code{get_observable_unit()} returns a character vector: column names
whose unique value combinations define the minimum observable unit.

\code{get_versioned()} returns a logical value: whether the dataset is
versioned using \code{as_of} dates.

\code{get_has_output_type_ids()} returns a logical value: whether
oracle-output data has \code{output_type} and \code{output_type_id} columns (default
\code{FALSE} if not specified).

\code{get_non_task_id_schema()} returns a named list: key-value pairs of
non-task ID column names and their data types, or \code{NULL} if not specified.

\code{has_target_data_config()} returns a logical value: \code{TRUE} if the
target-data.json file exists in the \code{hub-config} directory of the hub,
\code{FALSE} otherwise.
}
\description{
Utility functions for extracting properties from target-data.json configuration
files (v6.0.0 schema). These functions handle defaults and inheritance patterns
for target data configuration.
}
\details{
\subsection{Inheritance and Defaults}{

Some properties can be specified at both the global level and the dataset level:
\itemize{
\item \strong{observable_unit}: Dataset-specific values override global when specified,
otherwise the global value is used.
\item \strong{versioned}: Dataset-specific values override global when specified,
otherwise inherits from global (default \code{FALSE} if not specified anywhere).
}

Other properties are dataset-specific only:
\itemize{
\item \strong{has_output_type_ids}: Only for oracle-output dataset (default \code{FALSE})
\item \strong{non_task_id_schema}: Only for time-series dataset (default \code{NULL})
}
}
}
\section{Functions}{
\itemize{
\item \code{get_date_col()}: Get the name of the date column across hub data.

\item \code{get_observable_unit()}: Get observable unit column names. Returns
dataset-specific observable_unit if configured, otherwise falls back to global.

\item \code{get_versioned()}: Get whether target data is versioned for the
specified dataset. Returns dataset-specific setting if configured, otherwise
inherits from global (default \code{FALSE} if not specified).

\item \code{get_has_output_type_ids()}: Get whether oracle-output data has
output_type/output_type_id columns.

\item \code{get_non_task_id_schema()}: Get the schema for non-task ID columns in
time-series data.

\item \code{has_target_data_config()}: Check if target data config file exists in hub.

}}
\examples{
hub_path <- system.file("testhubs/v6/target_dir", package = "hubUtils")
config <- read_config(hub_path, "target-data")

# Get the date column name
get_date_col(config)

# Get observable unit (uses dataset-specific or falls back to global)
get_observable_unit(config, dataset = "time-series")
get_observable_unit(config, dataset = "oracle-output")

# Get versioned setting (inherits from global if not specified)
get_versioned(config, dataset = "time-series")

# Get oracle-output specific property
get_has_output_type_ids(config)

# Get time-series specific property
get_non_task_id_schema(config)

# Check if target data config exists
has_target_data_config(hub_path)
no_config_hub <- system.file("testhubs/v5/target_file/", package = "hubUtils")
has_target_data_config(no_config_hub)
}
