% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ddbs_filter.R
\name{ddbs_filter}
\alias{ddbs_filter}
\title{Performs spatial filter of two geometries}
\usage{
ddbs_filter(
  x,
  y,
  predicate = "intersects",
  conn = NULL,
  name = NULL,
  crs = NULL,
  crs_column = "crs_duckspatial",
  distance = NULL,
  overwrite = FALSE,
  quiet = FALSE
)
}
\arguments{
\item{x}{An \code{sf} spatial object. Alternatively, it can be a string with the
name of a table with geometry column within the DuckDB database \code{conn}.
Data is returned from this object.}

\item{y}{Y table with geometry column within the DuckDB database}

\item{predicate}{A geometry predicate function. Defaults to \code{intersects},
a wrapper of \code{ST_Intersects}. See details for other options.}

\item{conn}{A connection object to a DuckDB database. If \code{NULL}, the function
runs on a temporary DuckDB database.}

\item{name}{A character string of length one specifying the name of the table,
or a character string of length two specifying the schema and table
names. If \code{NULL} (the default), the function returns the result as an
\code{sf} object}

\item{crs}{The coordinates reference system of the data. Specify if the data
doesn't have a \code{crs_column}, and you know the CRS.}

\item{crs_column}{a character string of length one specifying the column
storing the CRS (created automatically by \code{\link{ddbs_write_vector}}).
Set to \code{NULL} if absent.}

\item{distance}{a numeric value specifying the distance for ST_DWithin. Units correspond to
the coordinate system of the geometry (e.g. degrees or meters)}

\item{overwrite}{Boolean. whether to overwrite the existing table if it exists. Defaults
to \code{FALSE}. This argument is ignored when \code{name} is \code{NULL}.}

\item{quiet}{A logical value. If \code{TRUE}, suppresses any informational messages.
Defaults to \code{FALSE}.}
}
\value{
An sf object or TRUE (invisibly) for table creation
}
\description{
Filters data spatially based on a spatial predicate
}
\details{
Spatial Join Predicates:

A spatial predicate is really just a function that evaluates some spatial
relation between two geometries and returns true or false, e.g., “does a
contain b” or “is a within distance x of b”. Here is a quick overview of the
most commonly used ones, taking two geometries a and b:
\itemize{
\item \code{"ST_Intersects"}: Whether a intersects b
\item \code{"ST_Contains"}: Whether a contains b
\item \code{"ST_ContainsProperly"}: Whether a contains b without b touching a's boundary
\item \code{"ST_Within"}: Whether a is within b
\item \code{"ST_Overlaps"}: Whether a overlaps b
\item \code{"ST_Touches"}: Whether a touches b
\item \code{"ST_Equals"}: Whether a is equal to b
\item \code{"ST_Crosses"}: Whether a crosses b
\item \code{"ST_Covers"}: Whether a covers b
\item \code{"ST_CoveredBy"}: Whether a is covered by b
\item \code{"ST_DWithin"}: x)	Whether a is within distance x of b
}
}
\examples{
\dontrun{
## load packages
library(duckspatial)
library(sf)

# create a duckdb database in memory (with spatial extension)
conn <- ddbs_create_conn(dbdir = "memory")

## read data
countries_sf <- st_read(system.file("spatial/countries.geojson", package = "duckspatial"))
argentina_sf <- st_read(system.file("spatial/argentina.geojson", package = "duckspatial"))

## store in duckdb
ddbs_write_vector(conn, countries_sf, "countries")
ddbs_write_vector(conn, argentina_sf, "argentina")

## filter countries touching argentina
ddbs_filter(conn = conn, "countries", "argentina", predicate = "touches")

## filter without using a connection
ddbs_filter(countries_sf, argentina_sf, predicate = "touches")
}
}
