% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/rem_dyadiccutoffhelp.R
\name{remstats_dyadcut}
\alias{remstats_dyadcut}
\title{A Helper Function to Assist Researchers in Finding Dyadic Weight Cutoff Values}
\usage{
remstats_dyadcut(halflife = 2, relationalWidth, exp_weight_form = FALSE)
}
\arguments{
\item{halflife}{A numerical value that is the halflife value to be used in the exponential weighting function (see details section). Preset to 2 (should be updated by the user based on substantive context).}

\item{relationalWidth}{The numerical value that corresponds to the time range for which the user specifies for temporal relevancy.}

\item{exp_weight_form}{TRUE/FALSE. TRUE indicates that the Lerner et al. (2013) exponential weighting function will be used (see the details section). FALSE indicates that the Lerner and Lomi (2020) exponential weighting function will be used (see the details section). Set to FALSE by default}
}
\value{
The dyadic weight cutoff based on user specified values.
}
\description{
\ifelse{html}{\href{https://lifecycle.r-lib.org/articles/stages.html#stable}{\figure{lifecycle-stable.svg}{options: alt='[Stable]'}}}{\strong{[Stable]}}

A user-helper function to assist researchers in finding the dyadic
cutoff value to compute sufficient statistics for relational event models based upon temporal dependency.
}
\details{
This function is specifically designed as a user-helper function to assist
researchers in finding the dyadic cutoff value for creating sufficient statistics
based upon temporal dependency. In other words, this function estimates a dyadic
cutoff value for relational relevance, that is, the minimum dyadic weight for past
events to be potentially relevant (i.e., to possibly have an impact) on the current
event. All non-relevant events (i.e., events too distant in the past from the
current event to be considered relevant, that is, those below the cutoff value)
will have a weight of 0. This cutoff value is based upon two user-specified
values: the events' halflife (i..e, \code{halflife}) and the relationally relevant event
or time span (i.e., \code{relationalWidth}). Ideally, both the values for \code{halflife} and
\code{relationalWidth} would be based on the researcher’s command of the relevant
substantive literature. Importantly, \code{halflife} and \code{relationalWidth} must be in
the same units of measurement (e.g., days). If not, the function will not return
the correct answer.

For example, let’s say that the user defines the \code{halflife} to be 15
days (i.e., two weeks) and the relationally relevant event or time
span (i.e., \code{relationalWidth}) to be 30 days (i.e., events that occurred
more than 1 month in the past are not considered relationally relevant
for the current event). The user would then specify \code{halflife} = 15 and \code{relationalWidth} = 30.

Following Lerner and Lomi (2020), the exponential weighting function in
relational event models is:
\deqn{w(s, r, t) = e^{-(t-t') \cdot \frac{ln(2)}{T_{1/2}} }}

Following Lerner et al. (2013), the exponential weighting function in
relational event models is:
\deqn{w(s, r, t) = e^{-(t-t') \cdot \frac{ln(2)}{T_{1/2}} } \cdot \frac{ln(2)}{T_{1/2}}}

In both of the above equations, \emph{s} is the current event sender, \emph{r} is the
current event receiver (target), \emph{t} is the current event time, \emph{t'} is the
past event times that meet the weight subset, and \eqn{T_{1/2}} is the halflife parameter.
The task of this function is to find the weight, \eqn{ w(s, r, t)}, that corresponds to the
time difference provided by the user.
}
\examples{
#To replicate the example in the details section:
# with the Lerner et al. 2013 weighting function
remstats_dyadcut(halflife = 15,
                 relationalWidth = 30,
                 exp_weight_form = TRUE)

# without the Lerner et al. 2013 weighting function
remstats_dyadcut(halflife = 15,
                 relationalWidth = 30,
                 exp_weight_form = FALSE)

# A result to test the function (should come out to 0.50)
remstats_dyadcut(halflife = 30,
                 relationalWidth = 30,
                 exp_weight_form = FALSE)


# Replicating Lerner and Lomi (2020):
#"We set T1/2 to 30 days so that an event counts as (close to) one in the very next instant of time,
#it counts as 1/2 one month later, it counts as 1/4 two months after the event, and so on. To reduce
#the memory consumption needed to store the network of past events, we set a dyadic weight to
#zero if its value drops below 0.01. If a single event occurred in some dyad this would happen after
#6.64×T1/2, that is after more than half a year." (Lerner and Lomi 2020: 104).

# Based upon Lerner and Lomi (2020: 104), the result should be around 0.01. Since the
# time values in Lerner and Lomi (2020) are in milliseconds, we have to change
# all measurements into milliseconds
remstats_dyadcut(halflife = (30*24*60*60*1000), #30 days in milliseconds
                relationalWidth = (6.64*30*24*60*60*1000), #Based upon the paper
                #using the Lerner and Lomi (2020) weighting function
                exp_weight_form = FALSE)


}
\references{
Lerner, Jürgen and Alessandro Lomi. 2020. “Reliability of relational event
model estimates under sampling: How to fit a relational event model to 360
million dyadic events.” \emph{Network Science} 8(1): 97-135.

Lerner, Jürgen, Margit Bussman, Tom A.B. Snijders, and Ulrik Brandes. 2013. "
Modeling Frequency and Type of Interaction in Event Networks."
\emph{The Corvinus Journal of Sociology and Social Policy} 4(1): 3-32.
}
\author{
Kevin A. Carson \href{mailto:kacarson@arizona.edu}{kacarson@arizona.edu}, Diego F. Leal \href{mailto:dflc@arizona.edu}{dflc@arizona.edu}
}
