% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/utils.R
\name{pack}
\alias{pack}
\title{Pack GP and DGP emulators into a bundle}
\usage{
pack(..., id = NULL)
}
\arguments{
\item{...}{a sequence or a list of emulators produced by \code{\link[=gp]{gp()}} or \code{\link[=dgp]{dgp()}}.}

\item{id}{an ID to be assigned to the bundle emulator. If an ID is not provided (i.e., \code{id = NULL}), a UUID (Universally Unique Identifier) will be automatically generated
and assigned to the emulator. Default to \code{NULL}.}
}
\value{
An S3 class named \code{bundle} to be used by \code{\link[=design]{design()}} for sequential designs. It has:
\itemize{
\item a slot called \code{id} that is assigned through the \code{id} argument.
\item \emph{N} slots named \verb{emulator1,...,emulatorN}, each of which contains a GP or DGP emulator, where \emph{N} is the number of emulators
that are provided to the function.
\item a slot called \code{data} which contains two elements \code{X} and \code{Y}. \code{X} contains \emph{N} matrices named \verb{emulator1,...,emulatorN} that are
training input data for different emulators. \code{Y} contains \emph{N} single-column matrices named \verb{emulator1,...,emulatorN} that are
training output data for different emulators.
}
}
\description{
This function packs GP emulators and DGP emulators into a \code{bundle} class for
sequential designs if each emulator emulates one output dimension of the underlying simulator.
}
\details{
See further examples and tutorials at \url{https://mingdeyu.github.io/dgpsi-R/}.
}
\examples{
\dontrun{

# load packages
library(lhs)
library(dgpsi)

# construct a function with a two-dimensional output
f <- function(x) {
 y1 = sin(30*((2*x-1)/2-0.4)^5)*cos(20*((2*x-1)/2-0.4))
 y2 = 1/3*sin(2*(2*x - 1))+2/3*exp(-30*(2*(2*x-1))^2)+1/3
 return(cbind(y1,y2))
}

# generate the initial design
X <- maximinLHS(10,1)
Y <- f(X)

# generate the validation data
validate_x <- maximinLHS(30,1)
validate_y <- f(validate_x)

# training a 2-layered DGP emulator with respect to each output with the global connection off
m1 <- dgp(X, Y[,1], connect = F)
m2 <- dgp(X, Y[,2], connect = F)

# specify the range of the input dimension
lim <- c(0, 1)

# pack emulators to form an emulator bundle
m <- pack(m1, m2)

# 1st wave of the sequential design with 10 iterations and the target RMSE of 0.01
m <- design(m, N = 10, limits = lim, f = f, x_test = validate_x, y_test = validate_y, target = 0.01)

# 2rd wave of the sequential design with additional 10 iterations and the same target
m <- design(m, N = 10, limits = lim, f = f, x_test = validate_x, y_test = validate_y, target = 0.01)

# draw sequential designs of the two packed emulators
draw(m, type = 'design')

# inspect the traces of RMSEs of the two packed emulators
draw(m, type = 'rmse')

# write and read the constructed emulator bundle
write(m, 'bundle_dgp')
m <- read('bundle_dgp')

# unpack the bundle into individual emulators
m_unpacked <- unpack(m)

# plot OOS validations of individual emulators
plot(m_unpacked[[1]], x_test = validate_x, y_test = validate_y[,1])
plot(m_unpacked[[2]], x_test = validate_x, y_test = validate_y[,2])
}
}
