% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/trend_advanced.R
\name{trend_advanced}
\alias{trend_advanced}
\title{Determine multiple linear trends.}
\usage{
trend_advanced(
  var1,
  infile1,
  var2,
  infile2,
  outfile,
  nc34 = 4,
  overwrite = FALSE,
  verbose = FALSE,
  nc1 = NULL,
  nc2 = NULL
)
}
\arguments{
\item{var1}{Name of NetCDF variable of the first data set (character).}

\item{infile1}{Filename of input NetCDF file. This may include the directory
(character).}

\item{var2}{Name of NetCDF variable of the second data set (character).}

\item{infile2}{Filename of input NetCDF file. This may include the directory
(character).}

\item{outfile}{Filename of output NetCDF file. This may include the directory
(character).}

\item{nc34}{NetCDF version of output file. If \code{nc34 = 3} the output file will be
in NetCDFv3 format (numeric). Default output is NetCDFv4.}

\item{overwrite}{logical; should existing output file be overwritten?}

\item{verbose}{logical; if TRUE, progress messages are shown}

\item{nc1}{Alternatively to \code{infile1} you can specify the input as an
object of class \code{ncdf4} (as returned from \code{ncdf4::nc_open}).}

\item{nc2}{Alternatively to \code{infile2} you can specify the input as an
object of class \code{ncdf4} (as returned from \code{ncdf4::nc_open}).}
}
\value{
A NetCDF file including four data layers is written. One layer
(trend1) contains the linear trend based on the time steps.
Another layer (trend2) contains linear trend based on var2.
The two other layers contain a measure for the significance of the
calculated trends, which was derived using the 95 \% confidence interval.
The significance is calculated from the lower and upper value of the
95\% confidence interval:
lower or upper value < 0: sig = 0 (not significant);
lower and upper value < 0: sig = -1 (negative significant);
lower and upper value > 0: sig = 1  (positive significant)
}
\description{
The function determines the trend from data of two CM SAF NetCDF input
files basing on a multiple linear model. Learn more
<http://www.sthda.com/english/articles/40-regression-analysis/
168-multiple-linear-regression-in-r/>
}
\examples{
## Create two example NetCDF files with a similar structure as used by CM
## SAF. The files are created with the ncdf4 package.  Alternatively
## example data can be freely downloaded here: <https://wui.cmsaf.eu/>

library(ncdf4)

## create some (non-realistic) example data
lon <- seq(10, 15, 0.5)
lat <- seq(50, 55, 0.5)
time <- as.Date("2000-05-31")
origin <- as.Date("1983-01-01 00:00:00")
time <- as.numeric(difftime(time, origin, units = "hour"))
data1 <- array(250:350, dim = c(11, 11, 1))
data2 <- array(230:320, dim = c(11, 11, 1))

## create example NetCDF
x <- ncdim_def(name = "lon", units = "degrees_east", vals = lon)
y <- ncdim_def(name = "lat", units = "degrees_north", vals = lat)
t <- ncdim_def(name = "time", units = "hours since 1983-01-01 00:00:00",
             vals = time, unlim = TRUE)
var1 <- ncvar_def("SIS", "W m-2", list(x, y, t), -999, prec = "float")
vars <- list(var1)
ncnew_1 <- nc_create(file.path(tempdir(), "CMSAF_example_file_1.nc"), vars)
ncnew_2 <- nc_create(file.path(tempdir(), "CMSAF_example_file_2.nc"), vars)

ncvar_put(ncnew_1, var1, data1)
ncvar_put(ncnew_2, var1, data2)

ncatt_put(ncnew_1, "lon", "standard_name", "longitude", prec = "text")
ncatt_put(ncnew_1, "lat", "standard_name", "latitude", prec = "text")

ncatt_put(ncnew_2, "lon", "standard_name", "longitude", prec = "text")
ncatt_put(ncnew_2, "lat", "standard_name", "latitude", prec = "text")
nc_close(ncnew_1)
nc_close(ncnew_2)

## Determine the multiple linear trend of the example CM SAF NetCDF files and
## write the output to a new file.
trend_advanced(var1 = "SIS", infile1 = file.path(tempdir(),"CMSAF_example_file_1.nc"), 
      var2 = "SIS", infile2 = file.path(tempdir(), "CMSAF_example_file_2.nc"),
      outfile = file.path(tempdir(),"CMSAF_example_file_trend_advanced.nc"))

unlink(c(file.path(tempdir(),"CMSAF_example_file_1.nc"), 
      file.path(tempdir(),"CMSAF_example_file_2.nc"),
      file.path(tempdir(),"CMSAF_example_file_trend_advanced.nc")))
}
\seealso{
Other temporal operators: 
\code{\link{cmsaf.detrend}()},
\code{\link{cmsaf.mk.test}()},
\code{\link{cmsaf.regres}()},
\code{\link{num_above}()},
\code{\link{num_below}()},
\code{\link{num_equal}()},
\code{\link{timavg}()},
\code{\link{timmax}()},
\code{\link{timmean}()},
\code{\link{timmin}()},
\code{\link{timpctl}()},
\code{\link{timsd}()},
\code{\link{timsum}()},
\code{\link{trend}()}
}
\concept{temporal operators}
