#' Generic Probability-Probability(P–P) Plot Function
#'
#' Generates a P–P (probability–probability) plot for any custom or built-in
#' probability distribution. The function compares the empirical probabilities
#' of the sample data with the theoretical probabilities computed from a
#' user-specified cumulative distribution function (CDF).
#'
#' @param sample A numeric vector of sample observations.
#' @param pfun A cumulative distribution function (CDF) corresponding to the
#'   theoretical distribution (e.g., \code{pnorm}, \code{pexp}, or a custom CDF).
#' @param params A named list of distribution parameters
#'   (e.g., \code{list(mean = 0, sd = 1)} or \code{list(alpha = 2, lambda = 1)}).
#' @param fit.line Logical; if \code{TRUE} (default), a red least-squares
#'   regression line is added to the plot. Also, displays the regression
#'   line equation and R² value on the plot.
#'
#' @details
#' The P–P plot is used to assess how closely the empirical distribution of
#' a dataset matches a specified theoretical distribution. The points should
#' ideally fall along the 45° reference line if the model fits well. 
#'
#' Requires user-defined function \code{'pfun'} for the CDF of the 
#' user-defined continuous distribution.
#'
#' Missing values in the sample are automatically removed with a warning.
#'
#' @return
#' This function returns no value; it produces a P–P plot.
#'
#' @examples
#' # Example 1: Exponential distribution
#' set.seed(123)
#' x <- rexp(100, rate = 2)
#' pp.plot(x, pexp, list(rate = 2))
#'
#' # Example 2: Customizing the fitted line
#' pp.plot(x, pexp, list(rate = 2),
#'               fit.line = TRUE)
#'
#' # Example 3: Without regression line
#' pp.plot(x, pexp, list(rate = 2), fit.line = FALSE)
#'
#' # Example 4: Display regression equation and R² value
#' pp.plot(x, pexp, list(rate = 2))
#'
#' # Example 5: For a user defined distribution
#' # Exponentiated Exponential Power (EEP) Distribution
#' # Data
#' x <- waiting
#' pp.plot(x,
#'       params = list(alpha=0.3407, lambda=0.6068, theta=7.6150),
#'       pfun = pgen.exp.power, fit.line=TRUE)
#' 
#' @export
pp.plot <- function(sample, pfun, params,
						  fit.line = TRUE) {
  # Input validation
  if (!is.numeric(sample) || length(sample) < 2)
    stop("'sample' must be a numeric vector with at least two observations.")
  if (!is.function(pfun))
    stop("'pfun' must be a valid cumulative distribution function.")
  if (!is.list(params) || is.null(names(params)))
    stop("'params' must be a named list of distribution parameters, 
          e.g. list(alpha = 1, lambda = 2).", call. = FALSE)

  na_count <- sum(is.na(sample))
  if (na_count > 0) warning(sprintf("Removed %d NA values.", na_count))
  sample <- na.omit(sample)

  n <- length(sample)
  sample <- sort(sample)
  emp_p <- ppoints(n)  # empirical probabilities
  theo_p <- do.call(pfun, c(list(q = sample), params))  # theoretical CDF values

  plot(theo_p, emp_p,
       main = "P-P Plot",
       xlab = "Theoretical Probabilities",
       ylab = "Empirical Probabilities",
       pch = 19, col = "blue", las = 1,
       xlim = c(0, 1), ylim = c(0, 1))

  abline(0, 1, col = "red", lty = 1, lwd = 2)

  if (isTRUE(fit.line)) {
    fit <- lm(emp_p ~ theo_p)
    abline(fit, col = "darkgreen", lwd = 2, lty = 2)

    # Show regression equation and R²
      coef_fit <- coef(fit)
      r2 <- summary(fit)$r.squared
      eq <- bquote(y == .(round(coef_fit[1], 3)) + 
                       .(round(coef_fit[2], 3)) * x ~~ 
                       (R^2 == .(round(r2, 4))))
      legend("topleft", legend = as.expression(eq), bty = "n", text.col = "darkgreen")
  }
}
