% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/pmt.R
\name{pmt}
\alias{pmt}
\alias{pmts}
\alias{define_pmt}
\title{Syntactic Sugar for Object Construction}
\usage{
pmt(key, ...)

pmts(
  which = c("all", "onesample", "twosample", "distribution", "association", "paired",
    "ksample", "multcomp", "rcbd", "table")
)

define_pmt(
  method = c("twosample", "distribution", "association", "paired", "ksample", "rcbd",
    "table"),
  statistic,
  rejection = c("<>", "<", ">"),
  scoring = "none",
  n_permu = 10000,
  name = "User-Defined Permutation Test",
  alternative = NULL,
  quickr = FALSE,
  depends = character(),
  plugins = character(),
  includes = character()
)
}
\arguments{
\item{key}{a character string specifying the test. Check \code{pmts()} for valid keys.}

\item{...}{extra parameters passed to the constructor.}

\item{which}{a character string specifying the desired tests.}

\item{method}{a character string specifying the permutation scheme.}

\item{statistic}{definition of the test statistic. See details.}

\item{rejection}{a character string specifying the rejection region relative to the test statistic.}

\item{scoring}{one of:
- a character string in \code{c("none", "rank", "vw", "expon")} specifying the scoring system
- a function that takes a numeric vector and returns an equal-length score vector}

\item{n_permu}{an integer indicating number of permutations for the permutation distribution. If set to \code{0}, all permutations will be used.}

\item{name, alternative}{character strings specifying the name of the test and the alternative hypothesis, used for printing purposes only.}

\item{quickr}{a logical indicating whether to use \code{\link[quickr:quick]{quickr::quick()}} to accelerate \code{statistic}. See details.}

\item{depends, plugins, includes}{passed to \code{\link[Rcpp:cppFunction]{Rcpp::cppFunction()}}.}
}
\value{
a test object corresponding to the specified key.

a data frame containing keys and corresponding tests implemented in this package.

a test object based on the specified statistic.
}
\description{
Construct test objects in a unified way.
}
\details{
The test statistic can be defined using either R or Rcpp, with the \code{statistic} parameter specified as:
\itemize{
\item \strong{R}: a closure returning one of
\itemize{
\item a double (the test statistic).
\item a closure returning a double.
}
\item \strong{Rcpp}: a character string defining a captureless lambda (since C++11) returning another lambda that captures by value, accepts parameters of the same type, and returns a double.
}

This design aims to pre-calculate potential constants that remain invariant during permutation.

When using Rcpp, the parameters for different \code{method} are listed as follows. Note that the names can be customized, and the types can be replaced with \code{auto} (thanks to the support for generic lambdas in C++14). See examples.\tabular{ccc}{
   \code{method} \tab Parameter 1 \tab Parameter 2 \cr
   \code{"twosample"} \tab \verb{const NumericVector& sample_1} \tab \verb{const NumericVector& sample_2} \cr
   \code{"distribution"} \tab \verb{const NumericVector& cumulative_prob_1} \tab \verb{const NumericVector& cumulative_prob_2} \cr
   \code{"association"} \tab \verb{const NumericVector& sample_1} \tab \verb{const NumericVector& sample_2} \cr
   \code{"paired"} \tab \verb{const NumericVector& sample_1} \tab \verb{const NumericVector& sample_2} \cr
   \code{"ksample"} \tab \verb{const NumericVector& combined_sample} \tab \verb{const IntegerVector& one_based_group_index} \cr
   \code{"rcbd"} \tab \verb{const NumericMatrix& block_as_column_data} \tab  \cr
   \code{"table"} \tab \verb{const IntegerMatrix& contingency_table} \tab  \cr
}


When using R, \code{statistic} and the parameters should be the R equivalents of the above. If no constants exist during permutation, \code{statistic} may simply be an R closure returning a double.

If \code{quickr = TRUE} and \code{statistic} returns a double, it will be compiled to Fortran via \code{\link[quickr:quick]{quickr::quick()}} with \code{\link[base:declare]{base::declare()}} calls for all arguments inserted automatically. Otherwise, \code{statistic} will be compiled using \code{\link[compiler:compile]{compiler::cmpfun()}}.
}
\note{
To improve performance when calling R closures from C++, this package repeatedly evaluates the closure's body in an environment whose enclosing environment is the closure's own, with its formal arguments pre-assigned to the data. This imposes the following restrictions on the closure returning the test statistic when \code{statistic} is written in R:
\itemize{
\item Do not re-assign its formal arguments or any pre-computed symbols in its environment.
\item Do not use default arguments or variadic arguments.
}

It's also worth noting that the data is permuted in-place. Therefore, modifications to the data within \code{statistic} may lead to incorrect results. It is recommended to avoid modifying the data when using R and pass const references as in the table above when using Rcpp.
}
\examples{
pmt("twosample.wilcoxon")

pmts("ksample")

x <- rnorm(5)
y <- rnorm(5, 1)

t <- define_pmt(
    method = "twosample", rejection = "<",
    scoring = base::rank, # equivalent to "rank"
    statistic = function(x, y) sum(x)
)$test(x, y)$print()

t$scoring <- function(x) qnorm(rank(x) / (length(x) + 1)) # equivalent to "vw"
t$print()

t$n_permu <- 0
t$print()

\donttest{
r <- define_pmt(
    method = "twosample", n_permu = 1e5,
    statistic = function(x, y) {
        m <- length(x)
        n <- length(y)
        function(x, y) sum(x) / m - sum(y) / n
    }
)

quickr <- define_pmt(
    method = "twosample", n_permu = 1e5, quickr = TRUE,
    statistic = function(x, y) sum(x) / length(x) - sum(y) / length(y)
)

rcpp <- define_pmt(
    method = "twosample", n_permu = 1e5,
    statistic = "[](const auto& x, const auto& y) {
        auto m = x.length();
        auto n = y.length();
        return [=](const auto& x, const auto& y) {
            return sum(x) / m - sum(y) / n;
        };
    }"
)

# equivalent
# rcpp <- define_pmt(
#     method = "twosample", n_permu = 1e5,
#     statistic = "[](const NumericVector& x, const NumericVector& y) {
#         R_xlen_t m = x.length();
#         R_xlen_t n = y.length();
#         return [m, n](const NumericVector& x, const NumericVector& y) -> double {
#             return sum(x) / m - sum(y) / n;
#         };
#     }"
# )

set.seed(0)
r$test(x, y)$print()
set.seed(0)
quickr$test(x, y)$print()
set.seed(0)
rcpp$test(x, y)$print()

options(LearnNonparam.pmt_progress = FALSE)
system.time(r$test(x, y))
system.time(quickr$test(x, y))
system.time(rcpp$test(x, y))
}

}
