findmleHMMnostarting <- function(J, x, obsdist, no.initials = 50, EM = FALSE, verbose = TRUE, seed = NULL, ...) {

  # Set seed if provided for reproducibility
  if (!is.null(seed)) {
    set.seed(seed)
  }

  # Validate input parameters
  if (!is.numeric(J) || floor(J) != J || J < 2) {
    stop("J must be an integer strictly greater than 1")
  }

  if (!obsdist %in% c("pois", "norm", "weibull", "zip", "nbinom", "zinb", "exp", "gamma", "lnorm", "gev", "ZInormal", "ZIgamma")) {
    stop("observation distribution is not supported")
  }

  # Warning for GEV distribution with direct optimization
  if (obsdist == "gev" && EM == FALSE) {
    old_warn <- getOption("warn")
    options(warn = 1)
    on.exit(options(warn = old_warn), add = TRUE)
    warning("Suggested to use the EM algorithm for the generalised extreme value distribution, this will take awhile")
  }

  # Initialize optimization tracking variables
  n <- length(x)
  iter <- 1
  min_loglik <- -1e300
  best_HMM <- NULL
  best_estimate <- NULL

  # Generate initial parameter estimates using clustering (except for GEV)
  if (obsdist != "gev") {
    start <- clusterHMM(J, x, obsdist, seed = seed)
  }

  # Special initialization for GEV distribution using sampling
  if (obsdist == "gev") {
    start <- list()
    start$obspar <- list()
    locls <- c()
    scalels <- c()
    shapels <- c()

    i <- 1
    attempts <- 0
    max_attempts <- 20000

    # Bootstrap GEV parameter estimates from random subsequences
    while (length(locls) < 1000 && attempts < max_attempts) {
      attempts <- attempts + 1
      tryCatch({
        startpoint <- runif(1, 1, n)
        length <- runif(1, 10, 100)
        if (startpoint + length > n) {
          data <- x[startpoint:length(n)]
        } else {
          data <- x[startpoint:(startpoint + length)]
        }

        estimate <- evd::fgev(data)$estimate
        if (!any(is.na(estimate)) && !any(is.infinite(estimate))) {
          locls[i] <- estimate[1]
          scalels[i] <- estimate[2]
          shapels[i] <- estimate[3]
          i <- i + 1
        }
      }, error = function(e) {}, warning = function(w) {})
    }

    # Warning if insufficient bootstrap samples obtained
    if (length(locls) < 1000) {
      warning(paste("Could only obtain", length(locls), "valid GEV estimates out of", attempts, "attempts"))
    }

    # Initialize parameter containers
    start$obspar$loc <- numeric(J)
    start$obspar$scale <- numeric(J)
    start$obspar$shape <- numeric(J)
  }

  # Helper function to generate new parameters
  generate_new_parameters <- function() {

    # Sample initial GEV parameters from bootstrap estimates
    if (obsdist == "gev") {
      for (j in 1:J) {
        start$obspar$loc[j] <- sample(locls, 1, replace = TRUE)
        start$obspar$scale[j] <- sample(scalels, 1, replace = TRUE)
        start$obspar$shape[j] <- sample(shapels, 1, replace = TRUE)
      }
    }

    # Generate random transition matrix with enhanced diagonal elements
    Pi <- matrix(runif(J * J, 0, 1), ncol = J)
    diag(Pi) <- diag(Pi) + rpois(1, 6) * apply(Pi, 1, sum)
    Pi <- Pi / matrix(rep(apply(Pi, 1, sum), each = J), nrow = J)

    # Generate random initial state distribution
    delta <- runif(J, 0, 1)
    delta <- delta / sum(delta)

    # Add random perturbations to initial parameter estimates
    current_obspar <- start$obspar
    if (!is.null(current_obspar)) {
      for (param_name in names(current_obspar)) {
        param_vector <- current_obspar[[param_name]]
        for (j in 1:length(param_vector)) {
          if (abs(param_vector[j]) > 1e-10) {
            range <- abs(param_vector[j]) / 4
            param_vector[j] <- param_vector[j] + runif(1, -range, range)

            if (obsdist != "gev") {
              if (param_name %in% c("lambda", "shape", "scale", "size", "mu", "rate", "sdlog", "sd") &&
                  param_vector[j] <= 0) {
                param_vector[j] <- abs(param_vector[j])
              }
            }

            if (obsdist == "gev") {
              if (param_name %in% c("scale") && param_vector[j] <= 0) {
                param_vector[j] <- abs(param_vector[j])
              }
            }

            if (param_name == "pi" && (param_vector[j] < 0 || param_vector[j] > 1)) {
              param_vector[j] <- max(0, min(1, param_vector[j]))
            }
          }
        }
        current_obspar[[param_name]] <- param_vector
      }
    }

    return(list(obspar = current_obspar, Pi = Pi, delta = delta))
  }

  # Main optimization loop with multiple random starts
  while (iter <= no.initials) {

    # Jitter seed each iteration to ensure different initial values
    if (!is.null(seed)) set.seed(seed + iter)

    success <- FALSE
    attempt_count <- 0
    max_attempts_per_iter <- 100

    # Keep trying until we get a successful fit for this iteration
    while (!success && attempt_count < max_attempts_per_iter) {
      attempt_count <- attempt_count + 1

      new_params <- generate_new_parameters()
      current_obspar <- new_params$obspar
      Pi <- new_params$Pi
      delta <- new_params$delta

      tryCatch({
        HMM <- findmleHMM(J = J, x = x, obsdist = obsdist,
                          obspar = current_obspar, Pi = Pi, EM = EM,
                          verbose = verbose, seed = seed, ...)
        current_loglik <- HMM$loglik

        # Update best model if current fit is better
        if (min_loglik <= current_loglik) {
          best_HMM <- HMM
          best_estimate <- HMM
          min_loglik <- current_loglik
        }

        if (verbose) message(paste("iteration=", iter, sep = ""))
        success <- TRUE

      }, error = function(e) {
        if (verbose) {
          message(paste("Error in attempt", attempt_count, "for iteration", iter, ":", e$message))
          if (attempt_count < max_attempts_per_iter) {
            message("Generating new parameters and trying again...")
          }
        }
      })
    }

    if (!success) {
      warning(paste("Failed to find valid parameters after", max_attempts_per_iter,
                    "attempts for iteration", iter, ". Skipping this iteration."))
    }

    iter <- iter + 1
  }

  HMM <- best_HMM
  return(HMM)
}


