confintHMM <- function(x, HMM, obsdist, level = 0.95, B = 100, EM = FALSE, verbose = TRUE, seed = NULL, ...) {
  # Set seed once at the beginning for reproducibility
  if (!is.null(seed)) {
    set.seed(seed)
  }

  # Extract model parameters
  n <- length(x)
  J <- length(HMM$estimate$delta)
  Pi <- HMM$estimate$Pi
  delta <- solve(t(diag(J) - Pi + 1), rep(1, J))
  obspar <- HMM$estimate[1:(length(HMM$estimate) - 2)]
  obs_names <- names(unlist(obspar))

  # Initialize bootstrap storage
  obs_boot <- matrix(NA, nrow = B, ncol = length(obs_names))
  colnames(obs_boot) <- obs_names
  Pi_boot <- array(NA, dim = c(B, J, J))

  # Bootstrap sampling with error handling
  successful_iter <- 0
  total_attempts <- 0

  while (successful_iter < B) {
    total_attempts <- total_attempts + 1

    if (verbose) {
      message(sprintf(
        "Bootstrap progress: %d / %d successful replicates (attempt %d)",
        successful_iter, B, total_attempts
      ))
    }

    # Generate bootstrap sample from fitted model
    # Note: No seed here - relies on RNG state set at function start
    sim <- generateHMM(n = n, J = J, obsdist = obsdist, obspar = obspar, Pi = Pi)
    x_boot <- sim$x

    # Fit HMM to bootstrap sample
    invisible(capture.output(fit <- tryCatch({
      findmleHMM(J = J, x = x_boot, obsdist = obsdist, obspar = obspar, Pi = Pi, EM = EM, verbose=verbose, ...)
    }, error = function(e) return(NULL))))

    # Store results if fitting was successful
    if (!is.null(fit) &&
        !is.null(fit$estimate) &&
        length(fit$estimate) >= 2 &&
        !is.null(fit$estimate$Pi)) {
      obs_params <- unlist(fit$estimate[1:(length(fit$estimate) - 2)])

      # Check for finite parameter estimates
      if (all(is.finite(obs_params)) && all(is.finite(fit$estimate$Pi))) {
        successful_iter <- successful_iter + 1
        obs_boot[successful_iter, ] <- obs_params
        Pi_boot[successful_iter, , ] <- fit$estimate$Pi
      }
    }

    # Prevent infinite loops for difficult cases
    if (total_attempts > B * 10) {
      warning(sprintf("Stopping after %d attempts. Only obtained %d valid replicates out of %d requested.",
                      total_attempts, successful_iter, B))
      break
    }
  }

  # Trim arrays if fewer than B successful iterations
  if (successful_iter < B) {
    obs_boot <- obs_boot[1:successful_iter, , drop = FALSE]
    Pi_boot <- Pi_boot[1:successful_iter, , , drop = FALSE]
  }

  # Compute confidence intervals using bootstrap quantiles
  alpha <- 1 - level

  # Observation parameter confidence intervals
  obs_ci <- t(apply(obs_boot, 2, quantile, probs = c(alpha / 2, 1 - alpha / 2)))
  colnames(obs_ci) <- c("Lower", "Upper")

  # Transition matrix confidence intervals
  Pi_lower <- matrix(NA, J, J)
  Pi_upper <- matrix(NA, J, J)
  for (j in 1:J) {
    for (k in 1:J) {
      Pi_vals <- Pi_boot[, j, k]
      Pi_lower[j, k] <- quantile(Pi_vals, probs = alpha / 2)
      Pi_upper[j, k] <- quantile(Pi_vals, probs = 1 - alpha / 2)
    }
  }

  # Format output for observation parameters
  obspar_CI <- data.frame(
    Parameter = colnames(obs_boot),
    Estimate = as.numeric(unlist(obspar)),
    Lower = obs_ci[, 1],
    Upper = obs_ci[, 2],
    row.names = NULL
  )

  # Return confidence intervals
  return(list(
    obspar_CI = obspar_CI,
    Pi_CI = list(
      Estimate = Pi,
      Lower = Pi_lower,
      Upper = Pi_upper
    ),
    bootstrap_samples = list(
      obspar = obs_boot,
      Pi = Pi_boot,
      n_successful = successful_iter,
      n_attempts = total_attempts
    )
  ))
}

