# ============================================================
# R/seasonal.R — Robust Seasonal Factors
# ============================================================

#' Internal helper: robust seasonal factor computation for GACE
#'
#' Produces stable multiplicative seasonal factors using:
#'  - robust seasonal medians,
#'  - trimmed ratios (0.70–1.30),
#'  - damping toward 1,
#'  - circular 3-point smoothing,
#'  - fallback for weak seasonality,
#'  - normalization to mean 1.
#'
#' @keywords internal
#' @noRd
#' @importFrom stats ts cycle median
.gace_compute_seasonal_factors <- function(y,
                                           freq_value,
                                           alpha = 0.5) {
  
  # -------------------------------------------------------------
  # No seasonality
  # -------------------------------------------------------------
  if (is.null(freq_value) || freq_value <= 1L) {
    return(rep(1.0, 1L))
  }
  
  # Ensure ts form
  if (!stats::is.ts(y)) {
    y_ts <- stats::ts(y, frequency = freq_value)
  } else {
    y_ts <- y
  }
  
  y_num <- as.numeric(y_ts)
  cyc   <- stats::cycle(y_ts)
  
  # -------------------------------------------------------------
  # Robust seasonal medians
  # -------------------------------------------------------------
  seas_mean <- tapply(y_num, cyc, stats::median, na.rm = TRUE)
  overall   <- stats::median(y_num, na.rm = TRUE)
  
  # Fallback
  if (!is.finite(overall) || overall <= 0) {
    return(rep(1.0, freq_value))
  }
  
  seas_mean[!is.finite(seas_mean)] <- overall
  
  # fix length mismatch
  seas_mean <- rep(seas_mean, length.out = freq_value)
  
  # ratios
  raw <- seas_mean / overall
  
  # -------------------------------------------------------------
  # Weak seasonality check — snap to 1 if near-flat
  # -------------------------------------------------------------
  if (sd(raw, na.rm = TRUE) < 1e-6) {
    return(rep(1.0, freq_value))
  }
  
  # -------------------------------------------------------------
  # Step 1: Trim to safe range
  # -------------------------------------------------------------
  raw <- pmax(pmin(raw, 1.30), 0.70)
  
  # -------------------------------------------------------------
  # Step 2: Damping toward 1
  # -------------------------------------------------------------
  damped <- 1 + alpha * (raw - 1)
  
  # -------------------------------------------------------------
  # Step 3: Circular smoothing (3-term moving average)
  # -------------------------------------------------------------
  if (freq_value > 2) {
    sm <- numeric(freq_value)
    for (i in seq_len(freq_value)) {
      left  <- if (i == 1) freq_value else (i - 1)
      right <- if (i == freq_value) 1 else (i + 1)
      sm[i] <- mean(damped[c(left, i, right)], na.rm = TRUE)
    }
    damped <- sm
  }
  
  # -------------------------------------------------------------
  # Step 4: Normalize to mean 1
  # -------------------------------------------------------------
  m <- mean(damped, na.rm = TRUE)
  if (!is.finite(m) || m <= 0) m <- 1
  
  damped <- damped / m
  
  # final guarantee
  damped[!is.finite(damped)] <- 1.0
  
  damped
}