package YouTube::Util;

our $AUTHORITY = 'cpan:PERLANCAR'; # AUTHORITY
our $DATE = '2020-11-26'; # DATE
our $DIST = 'YouTube-Util'; # DIST
our $VERSION = '0.003'; # VERSION

use 5.010001;
use strict;
use warnings;

use Regexp::Pattern::YouTube;

use Exporter qw(import);
our @EXPORT_OK = qw(
                       extract_youtube_video_id
               );

our %SPEC;

$SPEC{extract_youtube_video_id} = {
    v => 1.1,
    summary => 'Extract YouTube video ID from a string containing video ID, or youtube-dl-generated filename, or YouTube URL',
    description => <<'_',

Return undef if not found.

_
    args => {
        arg => {
            schema => ['str*'],
            req => 1,
            pos => 0,
        },
    },
    args_as => 'array',
    result_naked => 1,
    examples => [
        {args => {arg => 'https://www.youtube.com/watch?v=rp4UwPZfRis'}, result => 'rp4UwPZfRis'},
        {args => {arg => 'https://www.youtube.com/watch?v=cl1p7SOwnEk&list=PLjxOg4YY7Ah2JGb_QvA6KSB6YmikzA2fo&index=2&t=0s'}, result => 'cl1p7SOwnEk'},
        {args => {arg => 'https://www.youtube.com/watch?list=PLjxOg4YY7Ah2JGb_QvA6KSB6YmikzA2fo&v=cl1p7SOwnEk&index=2&t=0s'}, result => 'cl1p7SOwnEk'},
        {args => {arg => 'https://www.youtube.com/embed/U9v2S49sHeQ?rel=0'}, result => 'U9v2S49sHeQ'},
        {args => {arg => 'https://youtu.be/U9v2S49sHeQ'}, result => 'U9v2S49sHeQ'},
        {args => {arg => '$100,000 Name That Tune - Nick vs. Carol-SY-DnVZeFH0.mp4'}, result => 'SY-DnVZeFH0'},
        {args => {arg => 'GNdALsnBjh8'}, result => 'GNdALsnBjh8'},
        {args => {arg => 'foo'}, result => undef},
    ],
};
sub extract_youtube_video_id {
    my $arg = shift;

    my $re = $Regexp::Pattern::YouTube::RE{video_id}{pat};
    if ($arg =~ /\A$re\z/) {
        return $arg;
    } elsif ($arg =~ m!\Ahttps?://youtu\.be/($re)(?:\?|\#|\z)!) {
        return $1;
    } elsif ($arg =~ m!\Ahttps?://.+/embed/! &&
                 $arg =~ m!/embed/($re)(?:\?|\#|\z)!) {
        return $1;
    } elsif ($arg =~ m!\Ahttps?://! &&
                 $arg =~ /=($re)(?:&|\#|\z)/) {
        return $1;
    } else {
        # assume it's a filename like that generated by youtube-dl
        if ($arg =~ /-($re)(?:\.\w+)+(?:\s|\z)/) {
            return $1;
        }
    }
    undef;
}

1;
# ABSTRACT: YouTube-related utilities

__END__

=pod

=encoding UTF-8

=head1 NAME

YouTube::Util - YouTube-related utilities

=head1 VERSION

This document describes version 0.003 of YouTube::Util (from Perl distribution YouTube-Util), released on 2020-11-26.

=head1 FUNCTIONS


=head2 extract_youtube_video_id

Usage:

 extract_youtube_video_id($arg) -> any

Extract YouTube video ID from a string containing video ID, or youtube-dl-generated filename, or YouTube URL.

Examples:

=over

=item * Example #1:

 extract_youtube_video_id("https://www.youtube.com/watch?v=rp4UwPZfRis"); # -> "rp4UwPZfRis"

=item * Example #2:

 extract_youtube_video_id("https://www.youtube.com/watch?v=cl1p7SOwnEk&list=PLjxOg4YY7Ah2JGb_QvA6KSB6YmikzA2fo&index=2&t=0s");

Result:

 "cl1p7SOwnEk"

=item * Example #3:

 extract_youtube_video_id("https://www.youtube.com/watch?list=PLjxOg4YY7Ah2JGb_QvA6KSB6YmikzA2fo&v=cl1p7SOwnEk&index=2&t=0s");

Result:

 "cl1p7SOwnEk"

=item * Example #4:

 extract_youtube_video_id("https://www.youtube.com/embed/U9v2S49sHeQ?rel=0"); # -> "U9v2S49sHeQ"

=item * Example #5:

 extract_youtube_video_id("https://youtu.be/U9v2S49sHeQ"); # -> "U9v2S49sHeQ"

=item * Example #6:

 extract_youtube_video_id("\$100,000 Name That Tune - Nick vs. Carol-SY-DnVZeFH0.mp4"); # -> "SY-DnVZeFH0"

=item * Example #7:

 extract_youtube_video_id("GNdALsnBjh8"); # -> "GNdALsnBjh8"

=item * Example #8:

 extract_youtube_video_id("foo"); # -> undef

=back

Return undef if not found.

This function is not exported by default, but exportable.

Arguments ('*' denotes required arguments):

=over 4

=item * B<$arg>* => I<str>


=back

Return value:  (any)

=head1 HOMEPAGE

Please visit the project's homepage at L<https://metacpan.org/release/YouTube-Util>.

=head1 SOURCE

Source repository is at L<https://github.com/perlancar/perl-YouTube-Util>.

=head1 BUGS

Please report any bugs or feature requests on the bugtracker website L<https://rt.cpan.org/Public/Dist/Display.html?Name=YouTube-Util>

When submitting a bug or request, please include a test-file or a
patch to an existing test-file that illustrates the bug or desired
feature.

=head1 SEE ALSO

L<Regexp::Pattern::YouTube>

L<App::YouTubeUtils>

=head1 AUTHOR

perlancar <perlancar@cpan.org>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2020 by perlancar@cpan.org.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
